package ru.tinkoff.acquiring.sdk.redesign.payment.ui.v2

import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.models.PaymentSource
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.SavedCardsOptions
import ru.tinkoff.acquiring.sdk.models.paysources.AttachedCard
import ru.tinkoff.acquiring.sdk.models.paysources.CardSource
import ru.tinkoff.acquiring.sdk.payment.PaymentByCardProcess
import ru.tinkoff.acquiring.sdk.payment.PaymentByCardState
import ru.tinkoff.acquiring.sdk.redesign.common.emailinput.models.EmailValidator
import ru.tinkoff.acquiring.sdk.redesign.payment.model.CardChosenModel
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsAppBasedTransaction
import ru.tinkoff.acquiring.sdk.ui.customview.status.StatusViewData
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeResult
import ru.tinkoff.acquiring.sdk.ui.delegate.CardDataInputDelegate
import ru.tinkoff.acquiring.sdk.utils.BankCaptionProvider
import ru.tinkoff.acquiring.sdk.utils.NotificationFactory
import ru.tinkoff.acquiring.sdk.utils.lazyUnsafe

internal class PaymentBySavedCardViewModel(
    savedStateHandle: SavedStateHandle,
    private val paymentByCardProcess: PaymentByCardProcess,
    private val bankCaptionProvider: BankCaptionProvider,
    private val cardDataInputDelegate: CardDataInputDelegate,
) : AbstractPaymentByCardViewModel<PaymentBySavedCardViewModel.State, PaymentBySavedCardViewModel.Command>() {

    private val args: PaymentBySavedCardFragmentArgs =
        PaymentBySavedCardFragmentArgs.fromSavedStateHandle(savedStateHandle)

    private val savedCardsOptions: SavedCardsOptions by lazyUnsafe {
        SavedCardsOptions().apply {
            setTerminalParams(
                paymentOptions.terminalKey,
                paymentOptions.publicKey
            )
            customer = paymentOptions.customer
            features = paymentOptions.features
            mode = SavedCardsOptions.Mode.PAYMENT
            withArrowBack = true
        }
    }

    private val chosenCard = args.card.let {
        CardChosenModel(it, bankCaptionProvider(it.pan))
    }

    private val paymentOptions = args.paymentOptions

    private val _commandFlow = MutableSharedFlow<Command>()
    override val commandFlow = _commandFlow.asSharedFlow()

    private val _stateFlow: MutableStateFlow<State> = MutableStateFlow(createInitialState())
    override val stateFlow: StateFlow<State> = _stateFlow.asStateFlow()

    private fun createInitialState(): State {
        val email = paymentOptions.customer.email
        return State(
            showArrow = args.showArrow,
            chosenCardId = chosenCard.id,
            isValidEmail = email?.let { EmailValidator.validate(it) } ?: true,
            sendReceipt = email.isNullOrBlank().not(),
            email = email,
            paymentOptions = paymentOptions,
            chosenCard = chosenCard
        )
    }

    override fun validateCvc(cvc: String) {
        cardDataInputDelegate.validateCvc(
            cvc = cvc,
            isFocused = stateFlow.value.isCvcFocused,
            updateState = {cvc, isCvcValid ->
                _stateFlow.update {
                    it.copy(
                        cvc = cvc,
                        isValidCvc = isCvcValid
                    )
                }
            }
        )
    }

    private fun sendReceiptChanged(isSelect: Boolean) {
        _stateFlow.update {
            it.copy(
                sendReceipt = isSelect,
                isValidEmail = EmailValidator.validate(it.email)
            )
        }

        viewModelScope.launch {
            _commandFlow.emit(Command.ClearFocus)
        }
    }

    private fun onEmailChanged(email: String) {
        validateEmail(
            email = email,
            forceHideError = false
        )
    }

    fun processChallengeResult(result: AppBaseChallengeResult) = viewModelScope.launch {
        when (result) {
            AppBaseChallengeResult.Cancelled -> {
                _commandFlow.emit(Command.ReturnPaymentCancel)
            }

            is AppBaseChallengeResult.ProtocolError -> {
                onChallengeError(result.error)
            }

            is AppBaseChallengeResult.RuntimeError -> {
                onChallengeError(result.error)
            }

            is AppBaseChallengeResult.Error -> {
                onChallengeError(result.error)
            }

            is AppBaseChallengeResult.Success -> {
                returnSuccessResultOrShowNotification(
                    paymentId = result.paymentResult.paymentId,
                    cardId = result.paymentResult.cardId,
                    rebillId = result.paymentResult.rebillId
                )
            }

            is AppBaseChallengeResult.TimeOut -> {
                val onClick: () -> Unit = {
                    viewModelScope.launch {
                        _stateFlow.update {
                            it.copy(
                                notification = null
                            )
                        }
                        _commandFlow.emit(Command.ReturnPaymentError(result.error))
                    }
                }
                val notification = NotificationFactory.getPaymentTimeoutErrorNotification(
                    exception = result.error,
                    onClickFactory = { onClick }
                )
                _stateFlow.update {
                    it.copy(
                        notification = notification
                    )
                }
            }

            AppBaseChallengeResult.Loading -> {
                val notification = Notification(
                    type = StatusViewData.Type.PROGRESS,
                    title = R.string.acq_commonsheet_processing_title,
                    description = R.string.acq_commonsheet_processing_description,
                )
                _stateFlow.update {
                    it.copy(
                        notification = notification
                    )
                }
            }
        }
    }

    private fun onChallengeError(error: Throwable) {
        val onClick: () -> Unit = {
            viewModelScope.launch {
                _commandFlow.emit(Command.ReturnPaymentError(error))
            }
        }

        _stateFlow.update {
            it.copy(
                notification = NotificationFactory.getPaymentErrorNotification(
                    exception = error,
                    onClickFactory = { onClick },
                    options = NotificationFactory.getNotificationOptions(paymentOptions.sdkContext)
                )
            )
        }
    }

    private fun pay() {
        val stateValue = _stateFlow.value
        val emailForPayment = if (stateValue.sendReceipt) stateValue.email else null

        paymentByCardProcess.start(
            cardData = stateValue.cardSource,
            paymentOptions = stateValue.paymentOptions,
            email = emailForPayment
        )

        viewModelScope.launch {
            paymentByCardProcess.state.collectLatest {
                when (it) {
                    is PaymentByCardState.Started -> setProgress(true)
                    is PaymentByCardState.Created -> Unit
                    is PaymentByCardState.Error -> returnErrorOrShowNotification(it.throwable)
                    is PaymentByCardState.Success -> {
                        returnSuccessResultOrShowNotification(
                            paymentId = it.paymentId,
                            cardId = it.cardId,
                            rebillId = it.rebillId
                        )
                    }

                    is PaymentByCardState.ThreeDsUiNeeded -> {
                        viewModelScope.launch {
                            _commandFlow.emit(
                                Command.RequestThreeDs(
                                    paymentOptions = paymentOptions,
                                    data = it.threeDsState.data,
                                    paymentSource = _stateFlow.value.cardSource
                                )
                            )
                        }
                        goTo3ds()
                    }

                    is PaymentByCardState.ThreeDsInProcess -> setProgress(false)

                    is PaymentByCardState.ThreeDsAppBase -> {
                        viewModelScope.launch {
                            setProgress(true)
                            _commandFlow.emit(
                                Command.RequestThreeAppBaseChallenge(
                                    transaction = it.transaction,
                                    threeDsData = it.threeDsData,
                                )
                            )
                        }
                    }

                    else -> Unit
                }
            }
        }
    }

    private fun returnErrorOrShowNotification(error: Throwable) {
        val onClick = {
            _stateFlow.update {
                it.copy(notification = null)
            }
            returnErrorResult(error)
        }
        if (paymentOptions.features.showPaymentNotifications) {
            val notification = NotificationFactory.getPaymentErrorNotification(
                exception = error,
                onClickFactory = { onClick },
                options = NotificationFactory.getNotificationOptions(paymentOptions.sdkContext)
            )
            _stateFlow.update {
                it.copy(
                    notification = notification
                )
            }
        } else {
            onClick()
        }
    }

    private fun returnSuccessResultOrShowNotification(
        paymentId: Long,
        cardId: String?,
        rebillId: String?
    ) {
        val onCLick = {
            returnSuccessResult(
                paymentId = paymentId,
                cardId = cardId,
                rebillId = rebillId
            )
        }
        if (paymentOptions.features.showPaymentNotifications) {
            val notification =
                NotificationFactory.getPaymentSuccessCommonNotification(
                    amount = paymentOptions.order.amount,
                    onClick = onCLick
                )
            _stateFlow.update {
                it.copy(
                    notification = notification
                )
            }
        } else {
            onCLick()
        }
    }

    private fun returnSuccessResult(paymentId: Long, cardId: String?, rebillId: String?) {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.ReturnPaymentSuccess(
                    paymentId = paymentId,
                    cardId = cardId,
                    rebillId = rebillId
                )
            )
        }
    }

    private fun returnErrorResult(throwable: Throwable) {
        viewModelScope.launch {
            _commandFlow.emit(Command.ReturnPaymentError(throwable))
        }
    }

    private fun setProgress(progress: Boolean) {
        _stateFlow.update {
            it.copy(progress = progress)
        }
    }

    private fun goTo3ds() {
        paymentByCardProcess.goTo3ds()
    }

    private fun cancelPayment() {
        paymentByCardProcess.stop()
    }

    data class State(
        val showArrow: Boolean = false,
        val paymentOptions: PaymentOptions,
        val progress: Boolean = false,
        val notification: Notification? = null,
        val chosenCard: CardChosenModel,
        val chosenCardId: String? = null,
        val cvc: String? = null,
        val isValidCvc: Boolean? = null,
        val isCvcFocused: Boolean = false,
        val sendReceipt: Boolean = false,
        val email: String? = null,
        val isValidEmail: Boolean? = null,
        val isEmailFocused: Boolean = false,
    ) {

        val payButtonEnabled: Boolean
            get() {
                return if (sendReceipt) {
                    isValidEmail == true && isValidCvc == true
                } else {
                    isValidCvc == true
                }
            }

        val amount = paymentOptions.order.amount.toHumanReadableString()

        val cardSource: CardSource
            get() = AttachedCard(chosenCardId, cvc)
    }

    fun handleEvent(event: Event) {
        when (event) {
            Event.ChangeCard -> requestCardChange()
            Event.BackPressed -> onBackPressed()
            is Event.ChosenCardCvcChanged -> onCvcChanged(event.cvc)
            Event.PayButtonClick -> onPayButtonClick()
            is Event.SendReceiptChanged -> sendReceiptChanged(event.isChecked)
            Event.Resume -> onViewResume()
            is Event.EmailChanged -> onEmailChanged(event.email)
            is Event.EmailFocusChanged -> onEmailFocusChanged(event.isEmailFocused)
            is Event.CvcFocusChanged -> onCvcFocusChanged(event.cvcFocused)
        }
    }

    private fun onCvcFocusChanged(cvcFocused: Boolean) {
        val currentState = stateFlow.value
        _stateFlow.update {
            it.copy(
                isCvcFocused = cvcFocused
            )
        }
        if (currentState.isCvcFocused != cvcFocused) {
            validateCvc(currentState.cvc.orEmpty())
        }
    }

    private fun onEmailFocusChanged(emailFocused: Boolean) {
        val currentState = stateFlow.value
        _stateFlow.update {
            it.copy(isEmailFocused = emailFocused)
        }
        if (currentState.isEmailFocused != emailFocused) {
            validateEmail(
                email = currentState.email.orEmpty(),
                forceHideError = emailFocused
            )
        }
    }

    override fun onViewResume(isFirstTime: Boolean) {
        if (isFirstTime) {
            viewModelScope.launch {
                val email = _stateFlow.value.email
                if (!email.isNullOrBlank()) {
                    _commandFlow.emit(
                        Command.FillEmailField(
                            email = email,
                        )
                    )
                }

                _commandFlow.emit(Command.RequestCvcFieldFocus)
            }
        }
    }

    private fun validateEmail(email: String, forceHideError: Boolean) {
        var isValid: Boolean? = EmailValidator.validate(email)
        if (isValid == false && (!shouldDisplayEmailError(email) || forceHideError)) {
            isValid = null
        }
        _stateFlow.update {
            it.copy(
                email = email,
                isValidEmail = isValid
            )
        }
    }

    private fun shouldDisplayEmailError(email: String): Boolean {
        return !stateFlow.value.isEmailFocused
    }

    private fun onPayButtonClick() {
        viewModelScope.launch {
            _commandFlow.emit(Command.ClearFocus)
            pay()
        }
    }

    private fun onBackPressed() {
        cancelPayment()
        returnPaymentCancel()
    }

    private fun returnPaymentCancel() {
        viewModelScope.launch {
            _commandFlow.emit(Command.ReturnPaymentCancel)
        }
    }

    private fun requestCardChange() {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.ChangeCard(
                    savedCardsOptions = savedCardsOptions,
                    paymentOptions = paymentOptions
                )
            )
        }
    }

    sealed interface Event {
        data object Resume : Event
        class ChosenCardCvcChanged(
            val cvc: String
        ) : Event

        data object ChangeCard : Event
        data object BackPressed : Event
        data object PayButtonClick : Event

        class SendReceiptChanged(
            val isChecked: Boolean,
        ) : Event

        class EmailChanged(
            val email: String
        ) : Event

        class EmailFocusChanged(
            val isEmailFocused: Boolean
        ) : Event

        class CvcFocusChanged(
            val cvcFocused: Boolean
        ) : Event
    }

    sealed interface Command {
        class ReturnPaymentError(
            val throwable: Throwable
        ) : Command

        class RequestThreeDs(
            val paymentOptions: PaymentOptions,
            val data: ThreeDsData,
            val paymentSource: PaymentSource
        ) : Command

        class RequestThreeAppBaseChallenge(
            val transaction: ThreeDsAppBasedTransaction,
            val threeDsData: ThreeDsData,
        ) : Command

        class ReturnPaymentSuccess(
            val paymentId: Long,
            val cardId: String?,
            val rebillId: String?
        ) : Command

        class ChangeCard(
            val savedCardsOptions: SavedCardsOptions,
            val paymentOptions: PaymentOptions,
        ) : Command

        class FillEmailField(
            val email: String?
        ) : Command

        data object ReturnPaymentCancel : Command
        data object RequestEmailFieldFocus : Command
        data object RequestCvcFieldFocus : Command
        data object ClearFocus : Command

    }
}
