package ru.tinkoff.acquiring.sdk.redesign.recurrent

import android.content.Context
import android.content.Intent
import android.os.Parcelable
import androidx.activity.result.contract.ActivityResultContract
import kotlinx.parcelize.Parcelize
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.redesign.common.result.AcqPaymentResult
import ru.tinkoff.acquiring.sdk.redesign.main.SingleActivity
import ru.tinkoff.acquiring.sdk.redesign.main.SingleActivity.Result.RecurrentPaymentResult

object RecurrentPayLauncher {

    sealed class Result

    class Success(
        override val paymentId: Long,
        override val cardId: String? = null,
        override val rebillId: String? = null,
    ) : Result(), AcqPaymentResult.Success

    object Canceled : Result(), AcqPaymentResult.Canceled

    class Error(
        override val paymentId: Long?,
        override val error: Throwable,
        override val errorCode: Int?
    ) : Result(), AcqPaymentResult.Error

    @Parcelize
    class StartData(
        val card: Card,
        val paymentOptions: PaymentOptions,
    ) : Parcelable

    object Contract : ActivityResultContract<StartData, Result>() {
        override fun createIntent(context: Context, input: StartData): Intent {
            return SingleActivity.createIntent(
                context, SingleActivity.Arguments.RecurrentPaymentArguments(
                    options = input.paymentOptions,
                    card = input.card
                )
            )
        }

        override fun parseResult(resultCode: Int, intent: Intent?): Result {
            return when (val result: RecurrentPaymentResult? = SingleActivity.getResult(intent)) {
                is RecurrentPaymentResult.Success -> {
                    Success(
                        paymentId = result.paymentId,
                        rebillId = result.rebillId,
                    )
                }

                is RecurrentPaymentResult.Error -> {
                    Error(
                        paymentId = result.paymentId,
                        error = result.error,
                        errorCode = result.errorCode
                    )
                }

                else -> Canceled
            }
        }
    }
}
