package ru.tinkoff.acquiring.sdk.redesign.recurrent.presentation

import android.app.Application
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.createSavedStateHandle
import androidx.lifecycle.viewModelScope
import androidx.lifecycle.viewmodel.initializer
import androidx.lifecycle.viewmodel.viewModelFactory
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.TinkoffAcquiring
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.paysources.AttachedCard
import ru.tinkoff.acquiring.sdk.models.result.PaymentResult
import ru.tinkoff.acquiring.sdk.payment.PaymentByCardState
import ru.tinkoff.acquiring.sdk.payment.RecurrentPaymentProcess
import ru.tinkoff.acquiring.sdk.redesign.payment.model.CardChosenModel
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.redesign.recurrent.nav.RecurrentPaymentNavigation
import ru.tinkoff.acquiring.sdk.redesign.recurrent.ui.RecurrentPaymentEvent
import ru.tinkoff.acquiring.sdk.redesign.recurrent.ui.RecurrentPaymentFragmentArgs
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsDataCollector
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsHelper
import ru.tinkoff.acquiring.sdk.ui.customview.editcard.validators.CardValidator
import ru.tinkoff.acquiring.sdk.ui.customview.status.StatusViewData
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeResult
import ru.tinkoff.acquiring.sdk.utils.BankCaptionProvider
import ru.tinkoff.acquiring.sdk.utils.BankCaptionResourceProvider
import ru.tinkoff.acquiring.sdk.utils.NotificationFactory

internal class RecurrentPaymentViewModel(
    private val savedStateHandle: SavedStateHandle,
    private val recurrentPaymentProcess: RecurrentPaymentProcess,
    private val recurrentPaymentNavigation: RecurrentPaymentNavigation.Impl,
    private val bankCaptionProvider: BankCaptionProvider,
) : ViewModel(), RecurrentPaymentNavigation by recurrentPaymentNavigation {

    // started
    private val args = RecurrentPaymentFragmentArgs.fromSavedStateHandle(savedStateHandle)
    private val paymentOptions get() = args.paymentOptions
    private val card get() = args.card

    private val hideProgressNotification = !paymentOptions.features.showPaymentNotifications

    private val _stateFlow = MutableStateFlow(State(
        hideProgressNotification = hideProgressNotification
    ))
    val stateFlow = _stateFlow.asStateFlow()

    init {
        viewModelScope.launch {
            recurrentPaymentProcess.state.collectLatest {
                AcquiringSdk.log(it.toString())
                when (it) {
                    PaymentByCardState.CvcUiInProcess,
                    PaymentByCardState.ThreeDsInProcess,
                    PaymentByCardState.Created,
                    is PaymentByCardState.Started -> {
                        val notification = if (!hideProgressNotification)
                            Notification(
                                type = StatusViewData.Type.PROGRESS,
                                description = R.string.acq_commonsheet_processing_title
                            ) else null
                        _stateFlow.update {
                            it.copy(
                                isLoading = true,
                                isUiLocked = true,
                                notification = notification
                            )
                        }
                    }

                    is PaymentByCardState.Error -> {
                        _stateFlow.update {
                            it.copy(
                                isUiLocked = false
                            )
                        }
                        closeWithError(
                            error = it.throwable,
                            paymentId = it.paymentId
                        )
                    }

                    is PaymentByCardState.Success -> {
                        _stateFlow.update {
                            it.copy(
                                isUiLocked = false
                            )
                        }
                        closeWithSuccess(
                            rebillId = it.rebillId,
                            paymentId = it.paymentId
                        )
                    }

                    PaymentByCardState.Cancelled -> {
                        closeWithCancel()
                    }

                    is PaymentByCardState.ThreeDsUiNeeded -> {
                        _stateFlow.update {
                            it.copy(
                                isUiLocked = false
                            )
                        }
                        recurrentPaymentNavigation.eventChannel.trySend(
                            RecurrentPaymentEvent.To3ds(
                                it.paymentOptions,
                                it.threeDsState
                            )
                        )
                    }

                    is PaymentByCardState.CvcUiNeeded -> {
                        savedStateHandle[DATA_REJECTED_PAYMENT_ID] = it.rejectedPaymentId
                        _stateFlow.update {
                            val pan = checkNotNull(card.pan)
                            it.copy(
                                isLoading = false,
                                isUiLocked = false,
                                needCvcForCard = CardChosenModel(card, bankCaptionProvider(pan))
                            )
                        }
                    }

                    is PaymentByCardState.ThreeDsAppBase -> {
                        recurrentPaymentNavigation.eventChannel.trySend(
                            RecurrentPaymentEvent.StartChallenge(
                                it.transaction,
                                it.threeDsData,
                                it.paymentOptions,
                            )
                        )
                    }
                }
            }
        }
    }

    fun pay() {
        viewModelScope.launch {
            recurrentPaymentProcess.start(
                AttachedCard(card.rebillId),
                paymentOptions,
                paymentOptions.customer.email,
            )
        }
    }

    fun set3dsResult(error: Throwable?, paymentId: Long?) {
        recurrentPaymentProcess.set3dsResult(error, paymentId)
    }

    fun set3dsResult(paymentResult: PaymentResult) {
        recurrentPaymentProcess.set3dsResult(paymentResult, paymentOptions)
    }

    fun goTo3ds() {
        recurrentPaymentProcess.onThreeDsUiInProcess()
    }

    fun onClose() {
        viewModelScope.launch {
            when (val paymentState = recurrentPaymentProcess.state.value) {
                is PaymentByCardState.Error -> closeWithError(
                    error = paymentState.throwable,
                    paymentId = paymentState.paymentId
                )

                is PaymentByCardState.Success -> closeWithSuccess(
                    rebillId = paymentState.rebillId,
                    paymentId = paymentState.paymentId
                )

                is PaymentByCardState.CvcUiNeeded,
                PaymentByCardState.ThreeDsInProcess,
                is PaymentByCardState.ThreeDsUiNeeded -> closeWithCancel()

                is PaymentByCardState.Started,
                PaymentByCardState.Created -> Unit
                else -> Unit
            }
        }
    }

    private suspend fun closeWithCancel() {
        recurrentPaymentNavigation.eventChannel.send(
            RecurrentPaymentEvent.CloseWithCancel()
        )
    }

    fun proccessChallengeResult(appBaseChallengeResult: AppBaseChallengeResult) {
        recurrentPaymentProcess.setChallengeResult(appBaseChallengeResult, paymentOptions)
    }

    private suspend fun closeWithSuccess(rebillId: String?, paymentId: Long) {
        recurrentPaymentNavigation.eventChannel.send(
            RecurrentPaymentEvent.CloseWithSuccess(
                paymentId = paymentId,
                rebillId = rebillId,
            )
        )
    }

    private suspend fun closeWithError(error: Throwable, paymentId: Long?) {
        recurrentPaymentNavigation.eventChannel.send(
            RecurrentPaymentEvent.CloseWithError(
                throwable = error,
                paymentId = paymentId
            )
        )
    }

    private val cvc: StateFlow<String?> = savedStateHandle.getStateFlow(DATA_CVC, null)
    val cvcValid = cvc.map { CardValidator.validateSecurityCodeOrFalse(it) }
    private val rejectedPaymentId: StateFlow<String?> = savedStateHandle.getStateFlow(
        DATA_REJECTED_PAYMENT_ID, null
    )
    val needHideKeyboard = savedStateHandle.getStateFlow(DATA_NEED_HIDE_KEYBOARD, false)

    fun inputCvc(cvc: String) {
        savedStateHandle[DATA_CVC] = cvc
    }

    fun payRejected() {
        savedStateHandle[DATA_NEED_HIDE_KEYBOARD] = true
        recurrentPaymentProcess.startWithCvc(
            cvc = checkNotNull(cvc.value),
            rebillId = checkNotNull(card.rebillId),
            rejectedId = checkNotNull(rejectedPaymentId.value),
            paymentOptions = paymentOptions,
            email = paymentOptions.customer.email
        )
    }

    fun handleEvent(e: Event) {
        when(e) {
            is Event.ThreeDsLaunchError -> onThreeDsLauncherError(e.error)
        }
    }

    private fun onThreeDsLauncherError(error: Throwable) {
        _stateFlow.update {
            it.copy(
                notification = NotificationFactory.getPaymentErrorNotification(
                    exception = error,
                    onClickFactory = { ::onClose },
                    options = NotificationFactory.getNotificationOptions(paymentOptions.sdkContext)
                )
            )
        }
    }

    companion object {
        private const val DATA_CVC = "data_cvc"
        private const val DATA_REJECTED_PAYMENT_ID = "rejected_payment_id"
        private const val DATA_NEED_HIDE_KEYBOARD = "need_hide_keyboard"
    }

    sealed interface Event {
        class ThreeDsLaunchError(val error: Throwable) : Event
    }

    data class State(
        val notification: Notification? = null,
        val isUiLocked: Boolean = false,
        val isLoading: Boolean = false,
        val needCvcForCard: CardChosenModel? = null,
        val hideProgressNotification: Boolean = false
    )
}

fun getRecurrentViewModelsFactory(
    application: Application,
    paymentOptions: PaymentOptions,
    threeDsDataCollector: ThreeDsDataCollector = ThreeDsHelper.CollectData
) = viewModelFactory {
    RecurrentPaymentProcess.init(
        TinkoffAcquiring(
            application,
            paymentOptions.terminalKey,
            paymentOptions.publicKey
        ).sdk,
        application,
        threeDsDataCollector
    )
    val recurrentPaymentNavigation = RecurrentPaymentNavigation.Impl()
    val bankCaptionResourceProvider = BankCaptionResourceProvider(application)
    initializer {
        RecurrentPaymentViewModel(
            createSavedStateHandle(),
            RecurrentPaymentProcess.get(),
            recurrentPaymentNavigation,
            bankCaptionResourceProvider,
        )
    }
}
