package ru.tinkoff.acquiring.sdk.redesign.recurrent.ui

import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.WindowManager
import androidx.core.os.bundleOf
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.setFragmentResult
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.navigation.fragment.NavHostFragment
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.AcqRecurrentPaymentFlowFragmentBinding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.ui.delegate.NotificationUtils
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData

/**
 * @author s.y.biryukov
 */
class RecurrentPaymentFlowFragment : Fragment(), IsolatedKoinComponent {
    private var viewBinding: AcqRecurrentPaymentFlowFragmentBinding? = null
    private var navHostFragment: NavHostFragment? = null
    private val viewModel: RecurrentPaymentFlowViewModel by viewModel {
        parametersOf(
            SavedStateViewModelFactory(
                requireActivity().application, this, arguments
            )
        )
    }

    private var notificationBottomSheetBinding: NotificationUtils.NotificationBottomSheetBinding? =
        null

    override fun onCreateView(
        inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?
    ): View {
        val binding =
            AcqRecurrentPaymentFlowFragmentBinding.inflate(layoutInflater, container, false)
        viewBinding = binding
        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        this.notificationBottomSheetBinding = NotificationUtils.bindNotificationBottomSheet(
            activity = requireActivity()
        )

        val fragment = childFragmentManager.findFragmentById(R.id.main_content) as NavHostFragment
        navHostFragment = fragment

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.commandFlow.collect {
                    when (it) {
                        is RecurrentPaymentFlowViewModel.Command.OpenRecurrentPaymentScreen -> {
                            openRecurrentPaymentScreen(it)
                        }

                        is RecurrentPaymentFlowViewModel.Command.FinishWithResult -> {
                            finishWithResult(it.result)
                        }

                        RecurrentPaymentFlowViewModel.Command.CloseRecurrentPaymentScreen -> {
                            returnToEmptyScreen()
                        }
                    }
                }
            }
        }

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.stateFlow.collectLatest {
                    val notification = it.notification
                    if (notification == null) {
                        notificationBottomSheetBinding?.hide()
                    } else {
                        val data = notification.toStatusViewData(requireContext())
                        notificationBottomSheetBinding?.show(data)
                    }
                }
            }
        }

        RecurrentPaymentFragment.registerResultListener(fragment.childFragmentManager, this) {
            viewModel.handleEvent(RecurrentPaymentFlowViewModel.Event.Result(it))
        }
    }

    private fun returnToEmptyScreen() {
        navHostFragment?.navController?.popBackStack(R.id.emptyFragment, false)
    }

    private fun finishWithResult(result: RecurrentPaymentFlowViewModel.Result) {
        returnResult(result)
    }

    override fun onResume() {
        super.onResume()

        (requireActivity().window.attributes as WindowManager.LayoutParams).dimAmount = 0f

        viewModel.handleEvent(RecurrentPaymentFlowViewModel.Event.Resume)
    }

    private fun openRecurrentPaymentScreen(it: RecurrentPaymentFlowViewModel.Command.OpenRecurrentPaymentScreen) {
        navHostFragment?.navController?.navigate(
            RecurrentPaymentFragmentDirections.openRecurrentPayment(
                paymentOptions = it.paymentOptions,
                card = it.card
            )
        )
    }

    private fun returnResult(result: RecurrentPaymentFlowViewModel.Result) {
        setFragmentResult(
            RESULT_KEY, bundleOf(
                RESULT_DATA_KEY to result
            )
        )
    }

    companion object {
        internal const val ARG_RECURRENT_PAYMENT_OPTION = "ARG_RECURRENT_PAYMENT_OPTION"
        internal const val ARG_RECURRENT_PAYMENT_CARD = "ARG_RECURRENT_PAYMENT_CARD"
        private const val RESULT_KEY = "RecurrentPaymentFlowFragment.RESULT_KEY"
        private const val RESULT_DATA_KEY = "RecurrentPaymentFlowFragment.RESULT_DATA_KEY"

        fun newInstance(paymentOptions: PaymentOptions, card: Card): RecurrentPaymentFlowFragment {
            return RecurrentPaymentFlowFragment().apply {
                arguments = bundleOf(
                    ARG_RECURRENT_PAYMENT_OPTION to paymentOptions,
                    ARG_RECURRENT_PAYMENT_CARD to card
                )
            }
        }

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (RecurrentPaymentFlowViewModel.Result) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(RESULT_KEY, lifecycleOwner) { _, bundle ->
                val result = bundle.getParcelable<RecurrentPaymentFlowViewModel.Result>(RESULT_DATA_KEY)
                result?.let(listener)
            }
        }
    }
}
