package ru.tinkoff.acquiring.sdk.redesign.recurrent.ui

import android.os.Parcelable
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import kotlinx.parcelize.Parcelize
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.redesign.recurrent.ui.RecurrentPaymentFlowFragment.Companion.ARG_RECURRENT_PAYMENT_CARD
import ru.tinkoff.acquiring.sdk.redesign.recurrent.ui.RecurrentPaymentFlowFragment.Companion.ARG_RECURRENT_PAYMENT_OPTION
import ru.tinkoff.acquiring.sdk.utils.NotificationFactory

/**
 * @author s.y.biryukov
 */
class RecurrentPaymentFlowViewModel(
    savedStateHandle: SavedStateHandle
) : ViewModel() {
    private var isResumeFirstTime: Boolean = true
    private val _commandFlow = MutableSharedFlow<Command>()
    val commandFlow = _commandFlow.asSharedFlow()
    private val _stateFlow = MutableStateFlow(State())
    val stateFlow = _stateFlow

    val paymentOptions =
        requireNotNull(savedStateHandle.get<PaymentOptions>(ARG_RECURRENT_PAYMENT_OPTION)) {
            "get<PaymentOptions>(ARG_RECURRENT_PAYMENT_OPTION) is null"
        }
    val card: Card =
        requireNotNull(savedStateHandle.get<Card>(ARG_RECURRENT_PAYMENT_CARD)) {
            "get<Card>(ARG_RECURRENT_PAYMENT_CARD) is null"
        }

    fun handleEvent(event: Event) {
        when (event) {
            Event.Resume -> onViewResume()
            is Event.Result -> handleResult(event.value)
        }
    }

    private fun handleResult(value: RecurrentPaymentResult) {
        viewModelScope.launch {
            _commandFlow.emit(Command.CloseRecurrentPaymentScreen)
        }
        val result = when (value) {
            is RecurrentPaymentResult.CloseWithCancel -> Result.Cancelled
            is RecurrentPaymentResult.CloseWithError -> Result.FinishError(
                error = value.error,
                paymentId = value.paymentId
            )

            is RecurrentPaymentResult.CloseWithSuccess -> Result.Success(
                paymentId = value.paymentId,
                rebillId = value.rebillId
            )

            is RecurrentPaymentResult.CloseWithTimeOut -> Result.FinishError(
                error = value.error,
                paymentId = value.paymentId
            )
        }

        when (result) {
            is Result.FinishError -> {
                if (paymentOptions.features.showPaymentNotifications) {
                    val notification = NotificationFactory.getPaymentErrorNotification(
                        exception = result.error,
                        onClickFactory = {{ sendResult(result) }},
                        options = NotificationFactory.getNotificationOptions(paymentOptions.sdkContext)
                    )
                    _stateFlow.update {
                        it.copy(
                            notification = notification
                        )
                    }
                } else {
                    sendResult(result)
                }
            }

            is Result.Success -> {
                if (paymentOptions.features.showPaymentNotifications) {
                    _stateFlow.update {
                        it.copy(
                            notification = NotificationFactory.getPaymentSuccessCommonNotification(
                                amount = paymentOptions.order.amount
                            ) {
                                sendResult(result)
                            }
                        )
                    }
                } else {
                    sendResult(result)
                }
            }

            Result.Cancelled -> {
                sendResult(result)
            }
        }
    }

    private fun sendResult(result: Result) {
        viewModelScope.launch {
            _commandFlow.emit(Command.FinishWithResult(result))
        }
    }

    private fun onViewResume() {
        if (isResumeFirstTime) {
            isResumeFirstTime = false
            onResumeFirstTime()
        }
    }

    private fun onResumeFirstTime() {
        startRecurrentPayment()
    }

    private fun startRecurrentPayment() {
        viewModelScope.launch {
            _commandFlow.emit(Command.OpenRecurrentPaymentScreen(paymentOptions, card))
        }
    }

    data class State(
        val notification: Notification? = null
    )

    sealed interface Command {
        class OpenRecurrentPaymentScreen(
            val paymentOptions: PaymentOptions,
            val card: Card
        ) : Command

        data object CloseRecurrentPaymentScreen : Command

        class FinishWithResult(
            val result: Result
        ) : Command
    }

    sealed interface Event {
        class Result(val value: RecurrentPaymentResult) : Event

        data object Resume : Event
    }

    sealed interface Result : Parcelable {
        @Parcelize
        data object Cancelled : Result

        @Parcelize
        data class Success(
            val paymentId: Long,
            val rebillId: String? = null
        ) : Result

        @Parcelize
        data class FinishError(
            val error: Throwable,
            val paymentId: Long?,
        ) : Result
    }
}
