package ru.tinkoff.acquiring.sdk.redesign.recurrent.ui

import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.activity.addCallback
import androidx.core.os.bundleOf
import androidx.core.view.isVisible
import androidx.core.view.updatePadding
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.viewModels
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.navigation.fragment.navArgs
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.FragmentRecurrentPaymentBinding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.models.Card
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.models.result.PaymentResult
import ru.tinkoff.acquiring.sdk.redesign.common.cardpay.CardPayComponent
import ru.tinkoff.acquiring.sdk.redesign.mainform.ui.BottomSheetComponent
import ru.tinkoff.acquiring.sdk.redesign.recurrent.presentation.RecurrentPaymentViewModel
import ru.tinkoff.acquiring.sdk.redesign.recurrent.presentation.getRecurrentViewModelsFactory
import ru.tinkoff.acquiring.sdk.smartfield.AcqTextFieldView
import ru.tinkoff.acquiring.sdk.ui.activities.ThreeDsLauncher
import ru.tinkoff.acquiring.sdk.ui.customview.editcard.keyboard.SecureKeyboardController
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeDelegate
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeDelegateImpl
import ru.tinkoff.acquiring.sdk.utils.lazyUnsafe
import ru.tinkoff.acquiring.sdk.utils.requestCvcFocus
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData
import java.lang.ref.WeakReference

class RecurrentPaymentFragment : Fragment(),
    AppBaseChallengeDelegate by AppBaseChallengeDelegateImpl(), IsolatedKoinComponent {

    private lateinit var viewBinding: FragmentRecurrentPaymentBinding

    private val args: RecurrentPaymentFragmentArgs by navArgs()
    private val paymentOptions: PaymentOptions get() = args.paymentOptions

    private val factory by lazyUnsafe {
        getRecurrentViewModelsFactory(
            requireActivity().application,
            paymentOptions
        )
    }
    private val viewModel: RecurrentPaymentViewModel by viewModels { factory }

    private var bottomSheetComponent: BottomSheetComponent? = null

    private var cardPayComponent:CardPayComponent? = null

    private val threeDsBrowserBasedLauncher = registerForActivityResult(ThreeDsLauncher.Contract) {
        when (it) {
            is ThreeDsLauncher.Result.Success -> {
                viewModel.set3dsResult(it.result as PaymentResult)
            }

            is ThreeDsLauncher.Result.Error -> {
                viewModel.set3dsResult(it.error, it.paymentId)
            }

            ThreeDsLauncher.Result.Cancelled -> {
                viewModel.onClose()
            }
        }
    }
    private var keyboardStateListener: SecureKeyboardController.KeyboardStateListener? = null

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        bottomSheetComponent = BottomSheetComponent(
            root = viewBinding.root,
            sheet = viewBinding.acqRecurrentFormSheet,
            onClickOutside = {
                bottomSheetComponent?.hide()
                true
            },
            onSheetHidden = {
                viewModel.onClose()
            }
        )

        cardPayComponent = CardPayComponent(
            WeakReference(viewBinding.root),
            viewBinding = viewBinding.acqRecurrentFormPay,
            email = null,
            onCvcCompleted = viewModel::inputCvc,
            onPayClick = ::onPayClick,
            useSecureKeyboard = paymentOptions.features.useSecureKeyboard,
        )

        requireActivity().onBackPressedDispatcher.addCallback(viewLifecycleOwner) {
            viewModel.onClose()
        }

        updatePaymentState()
        updateCvcValidState()
        updateForceHideKeyboard()
        subscribeOnEvents()
        viewBinding.root.post {
            bottomSheetComponent?.onAttachedToWindow()
        }
        if (savedInstanceState == null) {
            viewModel.pay()
        }
    }

    override fun onDestroyView() {
        super.onDestroyView()
        bottomSheetComponent = null
        cardPayComponent = null
    }

    override fun onResume() {
        super.onResume()
        val instance = SecureKeyboardController.getInstance()
        instance.registerInsets(requireView())
        val listener = object : SecureKeyboardController.KeyboardStateListener {
            override fun onPaddingUpdated(height: Int, navigationHeight: Int): Boolean {
                val acqRecurrentFormSheet = viewBinding.acqRecurrentFormSheet
                acqRecurrentFormSheet.updatePadding(bottom = height)
                lifecycleScope.launch {
                    delay(100)
                    bottomSheetComponent?.trimSheetToContent(acqRecurrentFormSheet)
                }
                return true
            }
        }
        this.keyboardStateListener = listener
        instance.setKeyboardStateListener(listener)
    }

    override fun onPause() {
        super.onPause()
        this.keyboardStateListener = null
        SecureKeyboardController.getInstance().clear()
    }

    private fun onPayClick() {
        cardPayComponent?.clearCvcFocus()
        viewModel.payRejected()
    }

    private fun updatePaymentState() = viewLifecycleOwner.lifecycleScope.launch {
        repeatOnLifecycle(Lifecycle.State.RESUMED) {
            viewModel.stateFlow.collectLatest { state ->
                cardPayComponent?.renderLoader(state.isLoading)
                if (state.isUiLocked) {
                    bottomSheetComponent?.lock()
                } else {
                    bottomSheetComponent?.unlock()
                }

                if (state.needCvcForCard == null) {
                    if (state.hideProgressNotification) {
                        viewBinding.root.isVisible = false
                    }
                    viewBinding.acqRecurrentFormPayContainer.isVisible = false
                    cardPayComponent?.isVisible(false)

                    viewBinding.acqRecurrentStatusView.apply {
                        isVisible = true
                        if (state.notification != null) {
                            showNotification(state.notification.toStatusViewData(requireContext()))
                        }
                        bottomSheetComponent?.trimSheetToContent(this)
                    }
                } else {
                    if (state.hideProgressNotification) {
                        viewBinding.root.isVisible = true
                    }
                    viewBinding.acqRecurrentFormPayContainer.isVisible = true
                    cardPayComponent?.let {
                        it.isVisible(true)
                        it.renderInputCvc(state.needCvcForCard, paymentOptions)
                        if (!state.isLoading) {
                            requestCvcFocus(viewBinding.root, it)
                        }
                    }
                    viewBinding.acqRecurrentStatusView.isVisible = false
                    bottomSheetComponent?.trimSheetToContent(viewBinding.acqRecurrentFormPayContainer)
                }
            }
        }
    }

    private fun requestCvcFocus(root: ViewGroup, card: CardPayComponent) {
        root.findViewById<AcqTextFieldView>(R.id.cvc_input).requestCvcFocus(root, card)
    }

    private fun updateCvcValidState() = lifecycleScope.launch {
        viewModel.cvcValid.collectLatest {
            cardPayComponent?.renderEnable(it)
        }
    }

    private fun updateForceHideKeyboard() = lifecycleScope.launch {
        lifecycle.repeatOnLifecycle(Lifecycle.State.RESUMED) {
            viewModel.needHideKeyboard.filter { it }.collectLatest {
                cardPayComponent?.isKeyboardVisible(false)
            }
        }
    }

    private fun subscribeOnEvents() = lifecycleScope.launch {
        lifecycle.repeatOnLifecycle(Lifecycle.State.RESUMED) {
            viewModel.events.collectLatest {
                when (it) {
                    is RecurrentPaymentEvent.CloseWithCancel -> {
                        setResult(RecurrentPaymentResult.CloseWithCancel())
                    }

                    is RecurrentPaymentEvent.CloseWithError -> {
                        setResult(RecurrentPaymentResult.CloseWithError(
                            error = it.throwable,
                            paymentId = it.paymentId
                        ))
                    }

                    is RecurrentPaymentEvent.CloseWithSuccess -> {
                        setResult(
                            RecurrentPaymentResult.CloseWithSuccess(
                                it.paymentId,
                                it.rebillId
                            )
                        )
                    }

                    is RecurrentPaymentEvent.ChallengeTimeOut -> {
                        setResult(
                            RecurrentPaymentResult.CloseWithTimeOut(
                                it.paymentId,
                                it.error
                            )
                        )
                    }

                    is RecurrentPaymentEvent.To3ds -> {
                        tryLaunch3ds(it)
                    }

                    is RecurrentPaymentEvent.StartChallenge -> {
                        initAppBaseChallengeDelegate(
                            requireActivity(),
                            AcquiringSdk(
                                it.options.terminalKey,
                                it.options.publicKey,
                            ),
                            viewLifecycleOwner.lifecycleScope
                        )
                        launchChallenge(it.threeDsState, it.transaction, viewModel::proccessChallengeResult)
                    }
                }
            }
        }
    }

    private fun setResult(result: RecurrentPaymentResult) {
        parentFragmentManager.setFragmentResult(
            FRAGMENT_RESULT_KEY, bundleOf(
                FRAGMENT_RESULT_BUNDLE_KEY to result
            )
        )
    }

    private fun tryLaunch3ds(it: RecurrentPaymentEvent.To3ds) {
        try {
            threeDsBrowserBasedLauncher.launch(
                ThreeDsLauncher.Params(
                    data = it.threeDsState.data,
                    options = it.paymentOptions,
                    panSuffix = ""
                )
            )
        } catch (e: Throwable) {
            viewModel.handleEvent(RecurrentPaymentViewModel.Event.ThreeDsLaunchError(e))
        } finally {
            viewModel.goTo3ds()
        }
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        return FragmentRecurrentPaymentBinding.inflate(inflater, container, false)
            .also { this.viewBinding = it }
            .root
    }

    companion object {
        fun newInstance(paymentOptions: PaymentOptions, card: Card): Fragment {
            return RecurrentPaymentFragment().apply {
                arguments = RecurrentPaymentFragmentArgs(
                    paymentOptions = paymentOptions,
                    card = card,
                ).toBundle()
            }
        }

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (RecurrentPaymentResult) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(
                FRAGMENT_RESULT_KEY, lifecycleOwner
            ) { _, bundle ->
                val result =
                    bundle.getParcelable<RecurrentPaymentResult>(FRAGMENT_RESULT_BUNDLE_KEY)

                result?.let(listener)
            }
        }

        private val TAG = RecurrentPaymentFragment::class.simpleName
        internal val FRAGMENT_RESULT_KEY = "$TAG.FRAGMENT_RESULT_KEY"
        internal val FRAGMENT_RESULT_BUNDLE_KEY = "$TAG.FRAGMENT_RESULT_BUNDLE_KEY"
        internal const val ARG_RECURRENT_PAYMENT_OPTION = "ARG_RECURRENT_PAYMENT_OPTION"
    }
}

