package ru.tinkoff.acquiring.sdk.redesign.sbp.ui

import android.os.Bundle
import android.os.Parcelable
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.core.os.bundleOf
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.navigation.fragment.NavHostFragment
import kotlinx.coroutines.launch
import kotlinx.parcelize.Parcelize
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.AcqSbpPaymentFlowFragmentBinding
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions

/**
 * @author s.y.biryukov
 */
class SbpPaymentFlowFragment : Fragment(), IsolatedKoinComponent {

    private var viewBinding: AcqSbpPaymentFlowFragmentBinding? = null
    private var navHostFragment: NavHostFragment? = null
    private val viewModel: SbpPaymentFlowViewModel by viewModel {
        parametersOf(
            SavedStateViewModelFactory(
                requireActivity().application, this, arguments
            )
        )
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        val binding = AcqSbpPaymentFlowFragmentBinding.inflate(inflater, container, false)
        this.viewBinding = binding
        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        val fragment = childFragmentManager.findFragmentById(R.id.main_content) as NavHostFragment
        navHostFragment = fragment

        registerBankListFragmentResultListener(fragment.childFragmentManager)

        setFragmentNoBanksResultListener(fragment.childFragmentManager)

        subscribeOnCommands()
    }

    override fun onResume() {
        super.onResume()

        viewModel.handleEvent(SbpPaymentFlowViewModel.Event.Resume)
    }

    private fun subscribeOnCommands() = lifecycleScope.launch {
        repeatOnLifecycle(Lifecycle.State.STARTED) {
            viewModel.commandFlow.collect {
                when (it) {
                    SbpPaymentFlowViewModel.Command.ShowNoBanksPage -> gotoNoBanksPage()
                    SbpPaymentFlowViewModel.Command.CloseWithCancel -> returnResult(Result.Cancel)
                    is SbpPaymentFlowViewModel.Command.CloseWithError -> returnResult(
                        Result.Error(
                            error = it.error,
                        )
                    )

                    is SbpPaymentFlowViewModel.Command.CloseWithSuccess -> returnResult(
                        Result.Success(
                            paymentId = it.paymentId
                        )
                    )

                    is SbpPaymentFlowViewModel.Command.ShowBankListPage -> gotoBankListPage(it)
                }
            }
        }
    }


    private fun gotoBankListPage(command: SbpPaymentFlowViewModel.Command.ShowBankListPage) {
        navHostFragment?.navController?.navigate(
            SbpPaymentFragmentDirections.openSbpPayment(
                paymentOptions = command.paymentOptions
            )
        )
    }

    private fun registerBankListFragmentResultListener(fragmentManager: FragmentManager) {
        SbpPaymentFragment.registerResultListener(fragmentManager, this) {
            when (it) {
                is SbpPaymentResult.EmptyBanks -> {
                    viewModel.handleEvent(SbpPaymentFlowViewModel.Event.NoBanksLoaded)
                }

                is SbpPaymentResult.CloseWithCancel -> {
                    viewModel.handleEvent(SbpPaymentFlowViewModel.Event.PaymentCancel)
                }

                is SbpPaymentResult.CloseWithError -> {
                    viewModel.handleEvent(SbpPaymentFlowViewModel.Event.PaymentError(it.error))
                }

                is SbpPaymentResult.CloseWithSuccess -> {
                    viewModel.handleEvent(SbpPaymentFlowViewModel.Event.PaymentSuccess(it.paymentId))
                }
            }
        }
    }

    private fun gotoNoBanksPage() {
        navHostFragment?.navController?.navigate(SbpPaymentFragmentDirections.openNoBanksScreen())
    }

    private fun setFragmentNoBanksResultListener(fragmentManager: FragmentManager) {
        SbpNoBanksStubFragment.registerResultListener(
            fragmentManager, this
        ) {
            viewModel.handleEvent(SbpPaymentFlowViewModel.Event.NoBankPageClose)
        }
    }

    private fun returnResult(result: Result) {
        parentFragmentManager.setFragmentResult(
            REQUEST_KEY, bundleOf(
                RESULT_KEY to result
            )
        )
    }


    override fun onDestroyView() {
        super.onDestroyView()
        viewBinding = null
    }

    companion object {
        private const val REQUEST_KEY = "SbpPaymentFlowFragment.REQUEST_KEY"
        private const val RESULT_KEY = "SbpPaymentFlowFragment.RESULT_KEY"

        fun newInstance(paymentOptions: PaymentOptions): SbpPaymentFlowFragment {
            return SbpPaymentFlowFragment().apply {
                arguments = SbpPaymentFlowFragmentArgs(paymentOptions).toBundle()
            }
        }

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (Result) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(REQUEST_KEY, lifecycleOwner) { _, bundle ->
                bundle.getParcelable<Result>(RESULT_KEY)?.let(listener)
            }
        }
    }

    sealed interface Result : Parcelable {
        @Parcelize
        data object Cancel : Result

        @Parcelize
        data class Success(
            val paymentId: Long,
        ) : Result

        @Parcelize
        data class Error(
            val error: Throwable,
            val errorCode: Int? = null,
        ) : Result
    }
}
