package ru.tinkoff.acquiring.sdk.redesign.sbp.ui

import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions

/**
 * @author s.y.biryukov
 */
class SbpPaymentFlowViewModel(
    savedStateHandle: SavedStateHandle
) : ViewModel() {

    private var isFirstTimeResume: Boolean = true
    private val args: SbpPaymentFlowFragmentArgs = SbpPaymentFlowFragmentArgs.fromSavedStateHandle(savedStateHandle)

    private val _commandFlow = MutableSharedFlow<Command>()
    val commandFlow = _commandFlow.asSharedFlow()

    fun handleEvent(event: Event) {
        when (event) {
            Event.NoBanksLoaded -> onNoBanksLoaded()
            Event.PaymentCancel -> onPaymentCancel()
            is Event.PaymentError -> onPaymentError(event.error)
            is Event.PaymentSuccess -> onPaymentSuccess(event.paymentId)
            Event.Resume -> onResume()
            is Event.NoBankPageClose -> onNoBanksPageClose()
        }
    }

    private fun onNoBanksPageClose() {
        viewModelScope.launch {
            _commandFlow.emit(Command.CloseWithCancel)
        }
    }

    private fun onResume() {
        if (isFirstTimeResume) {
            isFirstTimeResume = false
            onResumeFirstTime()
        }
    }

    private fun onResumeFirstTime() {
        viewModelScope.launch {
            _commandFlow.emit(Command.ShowBankListPage(args.paymentOptions))
        }
    }

    private fun onPaymentError(error: Throwable) {
        viewModelScope.launch {
            _commandFlow.emit(Command.CloseWithError(error))
        }
    }

    private fun onPaymentSuccess(paymentId: Long) {
        viewModelScope.launch {
            _commandFlow.emit(Command.CloseWithSuccess(paymentId))
        }
    }

    private fun onPaymentCancel() {
        viewModelScope.launch {
            _commandFlow.emit(Command.CloseWithCancel)
        }
    }

    private fun onNoBanksLoaded() {
        viewModelScope.launch {
            _commandFlow.emit(Command.ShowNoBanksPage)
        }
    }

    sealed interface Event {
        data object NoBanksLoaded : Event
        data object PaymentCancel : Event
        data object Resume : Event
        class PaymentError(val error: Throwable) : Event
        class PaymentSuccess(val paymentId: Long) : Event
        data object NoBankPageClose : Event
    }

    sealed interface Command {
        data object ShowNoBanksPage : Command
        data object CloseWithCancel : Command
        class ShowBankListPage(val paymentOptions: PaymentOptions) : Command
        class CloseWithError(val error: Throwable) : Command
        class CloseWithSuccess(val paymentId: Long) : Command
    }
}
