package ru.tinkoff.acquiring.sdk.redesign.sbp.ui

import android.content.Context
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.ImageView
import android.widget.TextView
import androidx.activity.addCallback
import androidx.appcompat.app.AppCompatActivity
import androidx.core.os.bundleOf
import androidx.core.view.children
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.recyclerview.widget.RecyclerView
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.FragmentSbpPaymentBinding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.redesign.common.util.AcqShimmerAnimator
import ru.tinkoff.acquiring.sdk.redesign.common.util.openWebLink
import ru.tinkoff.acquiring.sdk.redesign.sbp.util.BankInfo
import ru.tinkoff.acquiring.sdk.redesign.sbp.util.SbpHelper
import ru.tinkoff.acquiring.sdk.ui.customview.status.StatusViewData
import ru.tinkoff.acquiring.sdk.ui.delegate.NotificationUtils
import ru.tinkoff.acquiring.sdk.utils.showById
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData

class SbpPaymentFragment : Fragment(), IsolatedKoinComponent {

    private var notificationBottomSheetBinding: NotificationUtils.NotificationBottomSheetBinding? =
        null
    private var viewBinding: FragmentSbpPaymentBinding? = null

    private var adapter: BankListAdapter = BankListAdapter()

    private val viewModel: SbpPaymentViewModel by viewModel {
        parametersOf(
            SavedStateViewModelFactory(requireActivity().application, this, arguments)
        )
    }


    override fun onAttach(context: Context) {
        super.onAttach(context)
        this.notificationBottomSheetBinding =
            NotificationUtils.bindNotificationBottomSheet(requireActivity())
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        return FragmentSbpPaymentBinding.inflate(inflater, container, false)
            .also { this.viewBinding = it }
            .root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        requireActivity().onBackPressedDispatcher.addCallback(viewLifecycleOwner) {
            viewModel.handleEvent(SbpPaymentViewModel.Event.BackPressed)
        }

        initToolbar()
        initViews()
        subscribeOnState()
    }

    override fun onResume() {
        super.onResume()
        viewModel.handleEvent(SbpPaymentViewModel.Event.Resume)
    }

    override fun onPause() {
        super.onPause()
        viewModel.handleEvent(SbpPaymentViewModel.Event.Pause)
    }

    override fun onDestroy() {
        super.onDestroy()
        (requireActivity() as? AppCompatActivity)?.setSupportActionBar(null)
    }

    private fun initToolbar() {
        val toolbar = viewBinding?.acqToolbar
        val activity = (requireActivity() as? AppCompatActivity)
        activity?.setSupportActionBar(toolbar)
        activity?.supportActionBar?.setDisplayHomeAsUpEnabled(true)
        activity?.supportActionBar?.setDisplayShowHomeEnabled(true)
        toolbar?.setTitle(R.string.acq_banklist_title)
        toolbar?.setNavigationOnClickListener {
            viewModel.handleEvent(SbpPaymentViewModel.Event.BackPressed)
        }
    }

    private fun initViews() {
        this.adapter = BankListAdapter()
        viewBinding?.acqBankListContent?.adapter = this.adapter
    }

    private fun subscribeOnState() {
        lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                subscribeOnUiState()
            }
        }

        lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.commandFlow.collect {
                    when (it) {
                        SbpPaymentViewModel.Command.ReturnCancelResult -> {
                            setResult(SbpPaymentResult.CloseWithCancel)
                        }

                        is SbpPaymentViewModel.Command.OpenBankApp -> {
                            SbpHelper.openSbpDeeplink(
                                it.bankInfo.deeplink,
                                requireActivity()
                            ) { error ->
                                viewModel.handleEvent(
                                    SbpPaymentViewModel.Event.BankAppOpenError(
                                        it.bankInfo,
                                        error
                                    )
                                )
                            }
                        }

                        is SbpPaymentViewModel.Command.ReturnErrorResult -> {
                            returnErrorResult(it.error)
                        }

                        is SbpPaymentViewModel.Command.ReturnSuccessResult -> {
                            returnSuccessResult(it.paymentId)
                        }

                        is SbpPaymentViewModel.Command.OpenBankSite -> {
                            openBankSite(it.url)
                        }

                        is SbpPaymentViewModel.Command.ReturnEmptyBanksResult -> {
                            setResult(SbpPaymentResult.EmptyBanks)
                        }
                    }
                }
            }
        }
    }

    private fun openBankSite(url: String) {
        requireActivity().openWebLink(url)
    }

    private suspend fun subscribeOnUiState() {
        viewModel.stateFlow.collectLatest {
            if (it.isBanksListLoadingProgress) {
                viewBinding?.acqViewFlipper?.showById(R.id.acq_bank_list_shimmer)
                viewBinding?.acqBankListShimmer?.root?.children?.toList()?.let { views ->
                    AcqShimmerAnimator.animateSequentially(views)
                }
                return@collectLatest
            }

            val stubNotification = it.stubNotification
            if (stubNotification != null) {
                showStub(stubNotification.toStatusViewData(requireContext()))
                return@collectLatest
            }

            val notification = it.notification
            if (notification == null) {
                notificationBottomSheetBinding?.hide()
            } else {
                val data = notification.toStatusViewData(requireContext())
                notificationBottomSheetBinding?.show(data)
            }


            viewBinding?.acqViewFlipper?.showById(R.id.acq_bank_list_content)
            adapter.setItems(it.banks)
        }
    }

    private fun setResult(result: SbpPaymentResult) {
        parentFragmentManager.setFragmentResult(
            FRAGMENT_RESULT_KEY, bundleOf(
                FRAGMENT_RESULT_BUNDLE_KEY to result
            )
        )
    }

    private fun onBankSelected(bank: BankInfo) {
        viewModel.handleEvent(SbpPaymentViewModel.Event.BankSelected(bank))
    }

    private fun showStub(data: StatusViewData) {
        viewBinding?.acqViewFlipper?.showById(R.id.acq_card_list_stub)

        viewBinding?.acqCardListStub?.fill(data)
    }

    private fun returnSuccessResult(paymentId: Long) {
        setResult(SbpPaymentResult.CloseWithSuccess(paymentId))
    }

    override fun onDestroyView() {
        viewBinding = null
        super.onDestroyView()
    }

    private fun returnErrorResult(throwable: Throwable) {
        setResult(SbpPaymentResult.CloseWithError(throwable))
    }

    inner class BankListAdapter : RecyclerView.Adapter<VH>() {
        private var banks: List<BankInfo> = emptyList()
        override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): VH =
            VH(
                LayoutInflater.from(requireContext()).inflate(
                    R.layout.acq_bank_list_item, parent, false
                )
            )

        override fun onBindViewHolder(holder: VH, position: Int) {
            val bank = banks.toList()[position]
            return holder.bind(bank)
        }

        override fun getItemCount(): Int = banks.size

        internal fun setItems(banks: List<BankInfo>) {
            if (this.banks != banks) {
                this.banks = banks
                notifyDataSetChanged()
            }
        }
    }

    inner class VH(view: View) : RecyclerView.ViewHolder(view) {

        private val logo = view.findViewById<ImageView>(R.id.acq_bank_list_item_logo)
        private val name = view.findViewById<TextView>(R.id.acq_bank_list_item_name)

        internal fun bind(bank: BankInfo) {
            logo.setImageDrawable(bank.appIcon)
            name.text = bank.appLabel

            itemView.setOnClickListener {
                onBankSelected(bank)
            }
        }
    }

    companion object {

        @JvmStatic
        fun newInstance(paymentOptions: PaymentOptions) =
            SbpPaymentFragment().apply {
                arguments = SbpPaymentFragmentArgs(
                    paymentOptions = paymentOptions
                ).toBundle()
            }

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (SbpPaymentResult) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(
                FRAGMENT_RESULT_KEY,
                lifecycleOwner
            ) { _, bundle ->
                val result: SbpPaymentResult? = bundle.getParcelable(FRAGMENT_RESULT_BUNDLE_KEY)
                result?.let(listener)
            }
        }

        private const val TAG = "SbpPaymentFragment"
        private const val FRAGMENT_RESULT_KEY = "$TAG.FRAGMENT_RESULT_KEY"
        private const val FRAGMENT_RESULT_BUNDLE_KEY = "$TAG.FRAGMENT_RESULT_BUNDLE_KEY"
    }
}
