package ru.tinkoff.acquiring.sdk.redesign.sbp.ui

import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.models.enums.ResponseStatus
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.payment.SbpPaymentProcess
import ru.tinkoff.acquiring.sdk.payment.SbpPaymentState
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.redesign.sbp.util.BankInfo
import ru.tinkoff.acquiring.sdk.redesign.sbp.util.NspkBankAppsResolver
import ru.tinkoff.acquiring.sdk.ui.customview.status.StatusViewData
import ru.tinkoff.acquiring.sdk.utils.ConnectionChecker
import ru.tinkoff.acquiring.sdk.utils.NotificationFactory

internal class SbpPaymentViewModel(
    savedStateHandle: SavedStateHandle,
    private val connectionChecker: ConnectionChecker,
    private val sbpPaymentProcess: SbpPaymentProcess,
    private val bankAppsResolver: NspkBankAppsResolver,
) : ViewModel() {

    private var isFirstResume: Boolean = true
    private val args = SbpPaymentFragmentArgs.fromSavedStateHandle(savedStateHandle)
    private val paymentOptions: PaymentOptions = args
        .paymentOptions

    private val _stateFlow = MutableStateFlow(State())
    val stateFlow = _stateFlow.asStateFlow()
    private val _commandFlow = MutableSharedFlow<Command>()
    val commandFlow = _commandFlow.asSharedFlow()

    init {
        viewModelScope.launch {
            sbpPaymentProcess.stateFlow.collectLatest {
                handlePaymentState(it)
            }
        }
    }

    private fun handlePaymentState(state: SbpPaymentState) {
        when (state) {
            SbpPaymentState.Created,
            is SbpPaymentState.Started -> Unit

            is SbpPaymentState.CheckingStatus -> {
                val status = state.status
                val notification = if (status != null && status == ResponseStatus.FORM_SHOWED) {
                    Notification(
                        type = StatusViewData.Type.PROGRESS,
                        description = R.string.acq_commonsheet_payment_waiting_title,
                        button = R.string.acq_commonsheet_payment_waiting_flat_button,
                        onClick = {
                            cancelPayment()
                        }
                    )
                } else {
                    Notification(
                        type = StatusViewData.Type.PROGRESS,
                        title = R.string.acq_commonsheet_processing_title,
                        description = R.string.acq_commonsheet_processing_description,
                        onClose = {
                            cancelPayment()
                        }
                    )
                }
                _stateFlow.update {
                    it.copy(notification = notification)
                }
            }

            is SbpPaymentState.GetBankListFailed -> {
                if (paymentOptions.features.showPaymentNotifications) {
                    _stateFlow.update {
                        it.copy(
                            isBanksListLoadingProgress = false,
                            stubNotification = NotificationFactory.getContentLoadingErrorNotification(
                                exception = state.throwable,
                                onClickFactory = { { onErrorButtonClick(state.throwable) } }
                            )
                        )
                    }
                } else {
                    closeWithError(error = state.throwable)
                }
            }

            is SbpPaymentState.LeaveOnBankApp -> {
                val notification = Notification(
                    type = StatusViewData.Type.PROGRESS,
                    description = R.string.acq_commonsheet_payment_waiting_title,
                    button = R.string.acq_commonsheet_payment_waiting_flat_button,
                    onClick = {
                        cancelPayment()
                    }
                )
                _stateFlow.update {
                    it.copy(
                        notification = notification,

                    )
                }
                viewModelScope.launch {
                    _commandFlow.emit(Command.OpenBankApp(state.bankInfo))
                }
            }

            is SbpPaymentState.NeedChooseOnUi -> {
                val banks = state.showApps.banks

                viewModelScope.launch {
                    _stateFlow.update {
                        it.copy(
                            isBanksListLoadingProgress = false,
                            banks = bankAppsResolver.getBanksInfo(banks)
                        )
                    }

                    if (banks.isEmpty()) {
                        _commandFlow.emit(Command.ReturnEmptyBanksResult)
                    }
                }
            }

            is SbpPaymentState.PaymentFailed -> {
                val onClick: () -> Unit = {
                    viewModelScope.launch {
                        closeNotification()
                        _commandFlow.emit(Command.ReturnErrorResult(
                            error = state.throwable
                        ))
                    }
                }
                if (paymentOptions.features.showPaymentNotifications) {
                    val notification = NotificationFactory.getPaymentErrorNotification(
                        exception = state.throwable,
                        onClickFactory = { onClick },
                        options = NotificationFactory.getNotificationOptions(paymentOptions.sdkContext)
                    )
                    _stateFlow.update {
                        it.copy(notification = notification)
                    }
                } else {
                    onClick()
                }
            }

            is SbpPaymentState.Success -> {
                val onClick: () -> Unit = {
                    viewModelScope.launch {
                        closeNotification()
                        _commandFlow.emit(Command.ReturnSuccessResult(state.paymentId))
                    }
                }
                if (paymentOptions.features.showPaymentNotifications) {
                    val notification = NotificationFactory.getPaymentSuccessCommonNotification(
                        amount = paymentOptions.order.amount,
                        onClick = onClick
                    )
                    _stateFlow.update {
                        it.copy(notification = notification)
                    }
                } else {
                    onClick()
                }
            }

            is SbpPaymentState.CheckingPaused,
            is SbpPaymentState.Stopped ->  Unit
        }
    }

    private fun closeNotification() {
        _stateFlow.update {
            it.copy(notification = null)
        }
    }

    private fun loadData() {
        val isOnline = connectionChecker.isOnline()
        if (!isOnline) {
            _stateFlow.update {
                it.copy(
                    stubNotification = NotificationFactory.getOfflineNotification { { loadData() } }
                )
            }
        } else {
            _stateFlow.update {
                it.copy(
                    stubNotification = null
                )
            }
        }
        if (isOnline) {
            _stateFlow.update {
                it.copy(isBanksListLoadingProgress = true)
            }
            sbpPaymentProcess.start(paymentOptions)
        }
    }

    private fun onGoingToBankApp(bankInfo: BankInfo) {
        sbpPaymentProcess.goingToBankApp(bankInfo)
    }

    private fun errorOpenBankApp(bankInfo: BankInfo, error: Throwable) {
        AcquiringSdk.log(error)
        if (bankInfo.weblink != null) {
            viewModelScope.launch {
                _commandFlow.emit(Command.OpenBankSite(bankInfo.weblink))
            }
        } else {
            if (paymentOptions.features.showPaymentNotifications) {
                _stateFlow.update {
                    it.copy(
                        notification = NotificationFactory.getErrorCommonNotification(
                            exception = error,
                            onClickFactory = { { onErrorButtonClick(error) } }
                        )
                    )
                }
            } else {
                closeWithError(error)
            }
            stopPaymentProcess()
        }
    }

    private fun startCheckingStatus() {
        sbpPaymentProcess.startCheckingStatus()
    }

    private fun cancelPayment() {
        stopPaymentProcess()
        closeNotification()
    }

    fun handleEvent(event: Event) {
        when (event) {
            Event.BackPressed -> onBackPressed()
            is Event.BankSelected -> {
                onGoingToBankApp(event.bankInfo)
            }

            is Event.BankAppOpenError -> errorOpenBankApp(
                event.bankInfo,
                event.error
            )
            Event.Resume -> onResume()
            Event.Pause -> onViewPause()
        }
    }

    private fun onViewPause() {
        sbpPaymentProcess.pause()
    }

    private fun onResume() {
        if (isFirstResume) {
            isFirstResume = false
            loadData()
        }
        startCheckingStatus()
    }

    private fun onErrorButtonClick(error: Throwable) {
        closeWithError(error)
    }

    private fun closeWithError(error: Throwable) {
        viewModelScope.launch {
            _commandFlow.emit(Command.ReturnErrorResult(error))
        }
    }

    private fun onBackPressed() {
        cancelPayment()
        viewModelScope.launch {
            _commandFlow.emit(Command.ReturnCancelResult)
        }
    }

    override fun onCleared() {
        super.onCleared()
        stopPaymentProcess()
    }

    private fun stopPaymentProcess() {
        sbpPaymentProcess.stop()
    }

    companion object {
        private const val TAG = "SbpPaymentViewModel"
    }

    data class State(
        val stubNotification: Notification? = null,
        val isBanksListLoadingProgress: Boolean = false,
        val banks: List<BankInfo> = emptyList(),
        val notification: Notification? = null,
    )

    sealed interface Event {
        class BankSelected(
            val bankInfo: BankInfo
        ) : Event
        data object BackPressed : Event
        class BankAppOpenError(
            val bankInfo: BankInfo,
            val error: Throwable
        ) : Event

        data object Resume : Event
        data object Pause : Event

    }

    sealed interface Command {
        data object ReturnCancelResult : Command
        class OpenBankApp(
            val bankInfo: BankInfo
        ) : Command

        class OpenBankSite(
            val url: String
        ) : Command

        class ReturnErrorResult(
            val error: Throwable
        ) : Command

        class ReturnSuccessResult(
            val paymentId: Long
        ) : Command

        data object ReturnEmptyBanksResult : Command
    }
}
