package ru.tinkoff.acquiring.sdk.redesign.sbp.util

import android.content.pm.PackageManager
import android.graphics.drawable.Drawable
import androidx.annotation.VisibleForTesting
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import ru.tinkoff.acquiring.sdk.utils.isPackageInstalled

internal interface NspkBankAppsResolver {
    suspend fun getBanksInfo(banks: List<NspkBank>): List<BankInfo>
}

data class BankInfo(
    val packageName: String,
    val deeplink: String,
    val weblink: String? = null,
    val appIcon: Drawable,
    val appLabel: String,
)

class NspkBankAppsResolverImpl(
    private var packageManager: PackageManager
) : NspkBankAppsResolver {
    override suspend fun getBanksInfo(banks: List<NspkBank>): List<BankInfo> =
        withContext(Dispatchers.Default) {
            banks.filter {
                    resolvePackageManager().isPackageInstalled(it.packageName)
                }
                .map {
                    BankInfo(
                        packageName = it.packageName,
                        deeplink = it.deepLink,
                        appIcon = getAppIcon(it.packageName),
                        appLabel = getAppLabel(it.packageName),
                        weblink = it.weblink
                    )
                }
        }

    private fun getAppLabel(packageName: String) = resolvePackageManager()
        .getApplicationLabel(resolvePackageManager().getApplicationInfo(packageName,0))
        .toString()

    private fun getAppIcon(packageName: String) =
        resolvePackageManager().getApplicationIcon(packageName)

    private fun resolvePackageManager() = (testPackageManager ?: packageManager)

    companion object {
        var testPackageManager: PackageManager? = null

        @VisibleForTesting
        fun replacePackageManager(packageManager: PackageManager?) {
            testPackageManager = packageManager
        }
    }
}
