package ru.tinkoff.acquiring.sdk.redesign.tpay

import android.content.Context
import android.content.Intent
import android.os.Parcelable
import androidx.activity.result.contract.ActivityResultContract
import kotlinx.parcelize.Parcelize
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.redesign.common.result.AcqPaymentResult
import ru.tinkoff.acquiring.sdk.redesign.main.SingleActivity
import ru.tinkoff.acquiring.sdk.redesign.main.SingleActivity.Result.TpayPaymentResult

object TpayLauncher {

    @Parcelize
    sealed class Result : Parcelable

    class Success(
        override val paymentId: Long,
        override val cardId: String? = null,
        override val rebillId: String? = null
    ) : Result(), AcqPaymentResult.Success, java.io.Serializable

    object Canceled : Result(), AcqPaymentResult.Canceled

    class Error(
        override val error: Throwable,
        override val errorCode: Int?,
        override val paymentId: Long?
    ) : Result(), AcqPaymentResult.Error, java.io.Serializable

    @Parcelize
    class StartData(
        val paymentOptions: PaymentOptions,
        val version: String,
    ) : Parcelable

    object Contract : ActivityResultContract<StartData, Result>() {
        override fun createIntent(context: Context, input: StartData): Intent =
            SingleActivity.createIntent(
                context, SingleActivity.Arguments.TpayPaymentArguments(
                    options = input.paymentOptions,
                    version = input.version
                )
            )

        override fun parseResult(resultCode: Int, intent: Intent?): Result {
            return when (val result: TpayPaymentResult? = SingleActivity.getResult(intent)) {
                is TpayPaymentResult.Success -> Success(
                    paymentId = result.paymentId,
                    cardId = result.cardId,
                    rebillId = result.rebillId
                )

                is TpayPaymentResult.Error -> {
                    Error(
                        error = result.error,
                        errorCode = result.errorCode,
                        paymentId = result.paymentId
                    )
                }

                else -> Canceled
            }
        }
    }
}
