package ru.tinkoff.acquiring.sdk.redesign.tpay.presentation

import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.payment.TpayPaymentState
import ru.tinkoff.acquiring.sdk.payment.TpayProcess
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.redesign.tpay.ui.TPayResult
import ru.tinkoff.acquiring.sdk.redesign.tpay.ui.TpayFragmentArgs
import ru.tinkoff.acquiring.sdk.ui.customview.status.StatusViewData
import ru.tinkoff.acquiring.sdk.utils.NotificationFactory

internal class TpayViewModel(
    savedStateHandle: SavedStateHandle,
    private val tpayProcess: TpayProcess
) : ViewModel() {

    private val args = TpayFragmentArgs.fromSavedStateHandle(savedStateHandle)
    private val paymentOptions: PaymentOptions = args.paymentOptions
    private val version: String = args.version

    private val _stateFlow = MutableStateFlow(State())
    val stateFlow = _stateFlow.asStateFlow()
    private val _commandFlow = MutableSharedFlow<Command>()
    val commandFlow = _commandFlow.asSharedFlow()
    private var isFirstTimeResume: Boolean = true

    init {
        viewModelScope.launch {
            tpayProcess.stateFlow.collect {
                handleTPayProcessState(it)
            }
        }
    }


    override fun onCleared() {
        tpayProcess.stop()
        super.onCleared()
    }

    private fun handleTPayProcessState(state: TpayPaymentState) {
        when (state) {
            is TpayPaymentState.CheckingStatus -> onCheckingStatus()
            TpayPaymentState.Created -> onPaymentCreated()
            is TpayPaymentState.Started -> onPaymentStarted()
            is TpayPaymentState.LeaveOnBankApp -> onLeaveOnBankApp()
            is TpayPaymentState.PaymentFailed -> onPaymentFailed(state)
            is TpayPaymentState.Success -> onPaymentSuccess(state)
            is TpayPaymentState.NeedChooseOnUi -> onNeedChooseOnUi(state)
            is TpayPaymentState.Stopped,
            is TpayPaymentState.Paused -> Unit
        }
    }

    private fun cancelPayment() {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.ReturnResult(TPayResult.Canceled)
            )
        }
    }

    private fun onNeedChooseOnUi(state: TpayPaymentState.NeedChooseOnUi) {
        viewModelScope.launch {
            _commandFlow.emit(Command.GoToTinkoff(state.deeplink))
        }
    }

    private fun onPaymentSuccess(state: TpayPaymentState.Success) {
        val onClick: () -> Unit = {
            viewModelScope.launch {
                _commandFlow.emit(
                    Command.ReturnResult(
                        TPayResult.Success(
                            paymentId = state.paymentId,
                            cardId = state.cardId,
                            rebillId = state.rebillId
                        )
                    )
                )
            }
        }

        if (paymentOptions.features.showPaymentNotifications) {
            val notification = NotificationFactory.getPaymentSuccessCommonNotification(
                amount = paymentOptions.order.amount,
                onClick = onClick
            )
            _stateFlow.update {
                it.copy(notification = notification)
            }
        } else {
            onClick()
        }
    }

    private fun onCheckingStatus() {
        showInProgressNotification()
    }

    private fun onLeaveOnBankApp() {
        showInProgressNotification()
    }

    private fun onPaymentStarted() {
        showInProgressNotification()
    }

    private fun onPaymentCreated() {
        showInProgressNotification()
    }

    private fun showInProgressNotification() {
        val notification = Notification(
            type = StatusViewData.Type.PROGRESS,
            description = R.string.acq_commonsheet_payment_waiting_title,
            button = R.string.acq_commonsheet_payment_waiting_flat_button,
            onClick = ::cancelPayment,
        )

        _stateFlow.update {
            it.copy(notification = notification)
        }
    }

    private fun onPaymentFailed(state: TpayPaymentState.PaymentFailed) {
        val throwable = state.throwable
        val onClick: () -> Unit = {
            viewModelScope.launch {
                _stateFlow.update {
                    it.copy(notification = null)
                }
                _commandFlow.emit(
                    Command.ReturnResult(
                        TPayResult.Error(
                            error = throwable,
                            errorCode = state.errorCode?.toIntOrNull(),
                            paymentId = state.paymentId
                        )
                    )
                )
            }
        }
        if (paymentOptions.features.showPaymentNotifications) {
            val notification = NotificationFactory.getPaymentErrorNotification(
                exception = throwable,
                onClickFactory = { onClick },
                options = NotificationFactory.getNotificationOptions(paymentOptions.sdkContext)
            )
            _stateFlow.update {
                it.copy(notification = notification)
            }
        } else {
            onClick()
        }
    }

    fun goingToBankApp() {
        tpayProcess.goingToBankApp()
    }

    fun handleEvent(event: Event) {
        when (event) {
            Event.BackPressed -> onBackPressed()
            Event.Resume -> onViewResume()
            Event.Pause -> onViewPause()
        }
    }

    private fun onResumeFirstTime() {
        tpayProcess.start(
            paymentOptions = paymentOptions,
            tpayVersion = version,
        )
    }

    private fun onViewResume() {
        if (isFirstTimeResume) {
            isFirstTimeResume = false
            onResumeFirstTime()
        }
        startCheckingStatus()
    }

    private fun onViewPause() {
        tpayProcess.pause()
    }

    private fun onBackPressed() {
        cancelPayment()
    }

    private fun startCheckingStatus() {
        tpayProcess.startCheckingStatus()
    }

    data class State(
        val notification: Notification? = null
    )

    sealed interface Command {
        class GoToTinkoff(val deeplink: String) : Command
        class ReturnResult(val result: TPayResult) : Command
    }

    sealed interface Event {
        data object BackPressed : Event
        data object Resume : Event
        data object Pause : Event
    }
}
