package ru.tinkoff.acquiring.sdk.redesign.tpay.ui

import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.activity.addCallback
import androidx.core.os.bundleOf
import androidx.core.view.WindowCompat
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.navigation.fragment.navArgs
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.databinding.FragmentTpayFlowBinding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.redesign.tpay.presentation.TpayViewModel
import ru.tinkoff.acquiring.sdk.redesign.tpay.util.TpayHelper.openTpayDeeplink
import ru.tinkoff.acquiring.sdk.ui.delegate.NotificationUtils
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData

class TpayFragment : Fragment(), IsolatedKoinComponent {

    private var notificationBottomSheetBinding: NotificationUtils.NotificationBottomSheetBinding? =
        null
    private var viewBinding: FragmentTpayFlowBinding? = null

    private val args: TpayFragmentArgs by navArgs()

    private val viewModel by viewModel<TpayViewModel> {
        parametersOf(
            SavedStateViewModelFactory(requireActivity().application, this, arguments)
        )
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        return FragmentTpayFlowBinding.inflate(inflater, container, false)
            .also { this.viewBinding = it }
            .root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        this.notificationBottomSheetBinding = NotificationUtils.bindNotificationBottomSheet(
            activity = requireActivity()
        )

        requireActivity().onBackPressedDispatcher.addCallback(viewLifecycleOwner) {
            viewModel.handleEvent(TpayViewModel.Event.BackPressed)
        }

        subscribeToState()
        subscribeOnCommands()
    }

    override fun onResume() {
        super.onResume()
        viewModel.handleEvent(TpayViewModel.Event.Resume)
    }

    override fun onPause() {
        super.onPause()
        viewModel.handleEvent(TpayViewModel.Event.Pause)
    }

    private fun subscribeToState() = lifecycleScope.launch {
        repeatOnLifecycle(Lifecycle.State.STARTED) {
            viewModel.stateFlow.collectLatest {
                val notification = it.notification
                if (notification == null) {
                    notificationBottomSheetBinding?.hide()
                } else {
                    notificationBottomSheetBinding?.show(
                        notification.toStatusViewData(requireContext())
                    )
                }
            }
        }
    }

    private fun setResult(result: TPayResult) {
        parentFragmentManager.setFragmentResult(
            FRAGMENT_RESULT_KEY, bundleOf(
                FRAGMENT_RESULT_BUNDLE_KEY to result
            )
        )
    }

    private fun subscribeOnCommands() = lifecycleScope.launch {
        repeatOnLifecycle(Lifecycle.State.STARTED) {
            viewModel.commandFlow.collect {
                when (it) {
                    is TpayViewModel.Command.ReturnResult -> {
                        setResult(it.result)
                    }

                    is TpayViewModel.Command.GoToTinkoff -> {
                        requireActivity().openTpayDeeplink(it.deeplink)
                        viewModel.goingToBankApp()
                    }
                }
            }
        }
    }

    override fun onDestroyView() {
        super.onDestroyView()
        viewBinding = null
    }

    companion object {

        @JvmStatic
        fun newInstance(paymentOptions: PaymentOptions, version: String) =
            TpayFragment().apply {
                arguments = TpayFragmentArgs(
                    paymentOptions = paymentOptions,
                    version = version,
                ).toBundle()
            }

        fun setFragmentResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (TPayResult) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(
                FRAGMENT_RESULT_KEY,
                lifecycleOwner
            ) { _, bundle ->

                val result =
                    bundle.getParcelable<TPayResult>(FRAGMENT_RESULT_BUNDLE_KEY)
                result?.let { listener.invoke(it) }
            }
        }

        private const val TAG = "TpayFragment"
        private const val FRAGMENT_RESULT_KEY = "$TAG.FRAGMENT_RESULT_KEY"
        private const val FRAGMENT_RESULT_BUNDLE_KEY = "$TAG.FRAGMENT_RESULT_BUNDLE_KEY"

    }
}
