package ru.tinkoff.acquiring.sdk.threeds

import android.content.Context
import android.util.Log
import com.google.gson.Gson
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import okhttp3.OkHttpClient
import okhttp3.Request
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsCertInfo.CertType.Companion.toWrapperType
import ru.tinkoff.core.components.threedswrapper.ThreeDSWrapper
import java.util.concurrent.TimeUnit

private const val CERTS_CONFIG_URL_TEST =
    "https://asdk-config-test.cdn-tinkoff.ru/certs-configs/asdk-certs-config.json"
private const val CERTS_CONFIG_URL_PROD =
    "https://asdk-config-prod.cdn-tinkoff.ru/certs-configs/asdk-certs-config.json"
private const val PREFS_NAME = "tinkoff_asdk_prefs"
private const val SDK_LAST_CERT_UPDATE = "SDK_LAST_CERT_UPDATE"
private const val TAG = "CertificateManager"

interface CertificateManager {

    suspend fun updateCertsConfigIfNeeded(certConfigUpdateInterval: Long): ThreeDsCertsConfig?

    suspend fun updateCertsIfNeeded(config: ThreeDsCertsConfig?, threeDSWrapper: ThreeDSWrapper)
}

class CertificateManagerImpl(
    private val context: Context
) : CertificateManager {

    private val okHttpClient = OkHttpClient()
    private val gson = Gson()

    private var certsConfig: ThreeDsCertsConfig? = null

    @Suppress("BlockingMethodInNonBlockingContext")
    override suspend fun updateCertsConfigIfNeeded(certConfigUpdateInterval: Long): ThreeDsCertsConfig? {
        if (certsConfig != null &&
            (System.currentTimeMillis() - getLastCertConfigUpdate()) < TimeUnit.MINUTES.toMillis(
                certConfigUpdateInterval
            )
        ) {
            return certsConfig
        }
        return withContext(Dispatchers.IO) {
            try {
                val request = Request.Builder().url(getCertsConfigUrl()).build()
                val response = okHttpClient.newCall(request).execute()
                val newConfig =
                    gson.fromJson(response.body?.charStream(), ThreeDsCertsConfig::class.java)
                putLastPreference(System.currentTimeMillis())
                Log.d(TAG, "newConfig ${newConfig.certsInfo}")
                if (newConfig != null) {
                    certsConfig = newConfig
                }
            } catch (ignored: Throwable) {
                Log.e(TAG, ignored.message.toString(), ignored)
            }
            certsConfig
        }
    }

    override suspend fun updateCertsIfNeeded(
        config: ThreeDsCertsConfig?,
        threeDSWrapper: ThreeDSWrapper
    ) {
        val checkCerts = threeDSWrapper.checkCerts()
        val certsUpdate = mutableListOf<ThreeDSWrapper.DsCertsUpdate>()
        config?.certsInfo?.forEach { certInfo ->
            val existing = checkCerts.find { it.dsId == certInfo.dsId }?.let {
                when (certInfo.type) {
                    ThreeDsCertInfo.CertType.DS -> it.dsCertInfo
                    ThreeDsCertInfo.CertType.CA -> it.caCertInfo
                }
            }
            if (existing?.certHash != certInfo.sha256Fingerprint || certInfo.forceUpdate) {
                certsUpdate.add(
                    ThreeDSWrapper.DsCertsUpdate(
                        certInfo.dsId, certInfo.type.toWrapperType(), certInfo.url
                    )
                )
            }
        }
        threeDSWrapper.updateCerts(certsUpdate)
    }

    private fun putLastPreference(time: Long) {
        val prefs = context.getSharedPreferences(PREFS_NAME, Context.MODE_PRIVATE)
        prefs.edit().putLong(SDK_LAST_CERT_UPDATE, time).apply()
    }

    private fun getLastCertConfigUpdate(): Long {
        val prefs = context.getSharedPreferences(PREFS_NAME, Context.MODE_PRIVATE)
        val lastCertUpdate = prefs.getLong(SDK_LAST_CERT_UPDATE, 0L)
        return lastCertUpdate
    }

    private fun getCertsConfigUrl() = when (AcquiringSdk.isDeveloperMode) {
        true -> CERTS_CONFIG_URL_TEST
        else -> CERTS_CONFIG_URL_PROD
    }
}