package ru.tinkoff.acquiring.sdk.threeds

import android.content.Context
import android.graphics.Point
import android.text.InputType.TYPE_CLASS_NUMBER
import android.util.Log
import android.view.WindowManager
import com.emvco3ds.sdk.spec.Transaction
import com.emvco3ds.sdk.spec.UiCustomization
import kotlinx.coroutines.runBlocking
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringSdkException
import ru.tinkoff.acquiring.sdk.utils.dpToPx
import ru.tinkoff.acquiring.sdk.utils.getColorString
import ru.tinkoff.core.components.threedswrapper.ThreeDSWrapper
import ru.tinkoff.core.components.threedswrapper.ThreeDSWrapper.Companion.cancelButtonCustomization
import ru.tinkoff.core.components.threedswrapper.ThreeDSWrapper.Companion.closeSafe
import ru.tinkoff.core.components.threedswrapper.ThreeDSWrapper.Companion.continueButtonCustomization
import ru.tinkoff.core.components.threedswrapper.ThreeDSWrapper.Companion.labelCustomization
import ru.tinkoff.core.components.threedswrapper.ThreeDSWrapper.Companion.nextButtonCustomization
import ru.tinkoff.core.components.threedswrapper.ThreeDSWrapper.Companion.resendButtonCustomization
import ru.tinkoff.core.components.threedswrapper.ThreeDSWrapper.Companion.setSdkAppId
import ru.tinkoff.core.components.threedswrapper.ThreeDSWrapper.Companion.submitButtonCustomization
import ru.tinkoff.core.components.threedswrapper.ThreeDSWrapper.Companion.textBoxCustomization
import ru.tinkoff.core.components.threedswrapper.ThreeDSWrapper.Companion.toolbarCustomization
import java.util.Locale
import java.util.TimeZone
import java.util.UUID
import java.util.concurrent.TimeUnit


private const val PREFS_NAME = "tinkoff_asdk_prefs"
private const val SDK_APP_ID_KEY = "sdk_app_id"
private const val FONT_NAME = "sans-serif"

private const val FONT_SIZE_TITLE = 17
private const val FONT_SIZE_NORMAL = 15
private const val FONT_SIZE_16 = 16
private const val FONT_SIZE_20 = 20
private const val CORNER_RADIUS = 16
private const val MARGIN_HORIZONTAL = 16f
private const val BORDER_SIZE = 10
private const val BUTTON_HEIGHT = 56

const val SDK_MAP_APP_ID_KEY = "sdkAppID"
const val SDK_ENC_DATA_KEY = "sdkEncData"
const val SDK_EPHEM_PUB_KEY = "sdkEphemPubKey"
const val SDK_MAX_TIMEOUT_KEY = "sdkMaxTimeout"
const val SDK_REFERENCE_NUMBER_KEY = "sdkReferenceNumber"
const val SDK_TRANS_ID_KEY = "sdkTransID"
const val SDK_INTERFACE_KEY = "sdkInterface"
const val SDK_UI_TYPE_KEY = "sdkUiType"

private const val TAG = "CreateAppBasedTransaction"

interface CreateAppBasedTransaction {

    suspend fun invoke(
        threeDsVersion: String,
        paymentSystem: String,
        data: MutableMap<String, String>
    ): ThreeDsAppBasedTransaction
}

class CreateAppBasedTransactionImpl(
    private val certificateManager: CertificateManager,
    private val context: Context
) : CreateAppBasedTransaction {

    /**
     * Максимальное время, в течение которого пользователь может подтвердить проведение транзакции
     * через 3DS app-based flow; задается в минутах, минимальное допустимое значение - 5 минут.
     */
    var maxTimeout = 5
        set(value) {
            field = value.coerceAtLeast(5)
        }

    /**
     * Минимальное время, через которое ASDK попытается снова обновить конфиг с актуальными
     * сертификатами для использования в 3DS SDK; задается в минутах.
     */
    var certConfigUpdateInterval = 240L

    /**
     * Map of Payment System to Directory Server ID values.
     */
    private var psToDsIdMap = mapOf(
        "visa" to "A000000003",
        "mir" to "A000000658",
        "mc" to "A000000004",
        "mock" to "mock",
        "upi" to "A000000333"
    )

    @Throws(Throwable::class)
    override suspend operator fun invoke(
        threeDsVersion: String,
        paymentSystem: String,
        data: MutableMap<String, String>
    ): ThreeDsAppBasedTransaction {
        val wrapper = initWrapper()
        val transaction = initTransaction(wrapper, threeDsVersion, paymentSystem, data)
        return ThreeDsAppBasedTransaction(wrapper, transaction)
    }

    private suspend fun initWrapper(): ThreeDSWrapper {
        return runBlocking {
            val wrapper = ThreeDSWrapper(
                when (AcquiringSdk.isDeveloperMode) {
                    true -> ThreeDSWrapper.EmbeddedCertsInfo.TEST
                    else -> ThreeDSWrapper.EmbeddedCertsInfo.PROD
                }
            ).init(context, ThreeDSWrapper.newConfigParameters {
                setSdkAppId(getSdkAppId().toString())
            }, Locale.getDefault().toString(), customize(context), ThreeDsLogger())

            wrapper.clearCachedCerts()

            val result = wrapper.updateCerts(
                ThreeDSWrapper.DsCertsUpdate(
                    ThreeDSWrapper.DsId.MIR,
                    ThreeDSWrapper.CertType.DS,
                    "https://ca-3ds2-test.cdn-tinkoff.ru/certs/mirSDK.cer"
                ),
                ThreeDSWrapper.DsCertsUpdate(
                    ThreeDSWrapper.DsId.MIR,
                    ThreeDSWrapper.CertType.CA,
                    "https://ca-3ds2-test.cdn-tinkoff.ru/certs/mirRootCA.cer"
                )
            )

            val config = certificateManager.updateCertsConfigIfNeeded(certConfigUpdateInterval)
            certificateManager.updateCertsIfNeeded(config, wrapper)
            config?.certsInfo?.let { psToDsIdMap = it.mapPsToDsId() }
            config?.certCheckInterval?.toLongOrNull()?.let { certConfigUpdateInterval = it }
            return@runBlocking wrapper
        }
    }

    private fun customize(context: Context): UiCustomization {
        val textColorString = context.getColorString(R.color.acq_const_black_lighter_20)
        val actionColor = context.getColorString(R.color.acq_const_blue)
        val colorAccent = context.getColorString(R.color.acq_const_yellow)
        val inputBackground = context.getColorString(R.color.acq_const_cool_gray_alpha_3)
        val background = context.getColorString(R.color.acq_const_white)
        val customization = ThreeDSWrapper.newUiCustomization {
            showChallengeInfoTextIndicator = false
            challengeInfoTextColor = textColorString
            backgroundColor = background
            val margin: Int = context.dpToPx(MARGIN_HORIZONTAL).toInt()
            val radius = context.dpToPx(CORNER_RADIUS)
            labelCustomization {
                headingTextColor = textColorString
                headingTextColor = textColorString
                headingTextFontSize = FONT_SIZE_TITLE
                headingTextFontName = FONT_NAME
                textColor = textColorString
                textFontSize = FONT_SIZE_NORMAL
                textFontName = FONT_NAME
            }
            textBoxCustomization {
                borderColor = background
                borderWidth = BORDER_SIZE
                backgroundColor = inputBackground
                backgroundColorDark = inputBackground
                disableLine = true
                inputType = TYPE_CLASS_NUMBER
                cornerRadius = radius
                cursorColor = actionColor
                textColor = textColorString
                textFontSize = FONT_SIZE_TITLE
                textFontName = FONT_NAME
                height = BUTTON_HEIGHT
            }
            toolbarCustomization {
                backgroundColor = background
                headerText = context.getString(R.string.acq_payments_threeds_accept_auth)
                buttonText = context.getString(R.string.acq_payments_threeds_cancel_auth)
                textColor = textColorString
                textFontSize = FONT_SIZE_20
                textFontName = FONT_NAME
            }
            submitButtonCustomization {
                backgroundColor = colorAccent
                cornerRadius = CORNER_RADIUS
                textColor = context.getColorString(R.color.acq_const_black_lighter_20)
                textFontSize = FONT_SIZE_TITLE
                textFontName = FONT_NAME
                height = BUTTON_HEIGHT
            }
            continueButtonCustomization {
                backgroundColor = colorAccent
                cornerRadius = CORNER_RADIUS
                textColor = textColorString
                textFontSize = FONT_SIZE_TITLE
                textFontName = FONT_NAME
                leftMargin = margin
                rightMargin = margin
            }
            nextButtonCustomization {
                backgroundColor = colorAccent
                cornerRadius = CORNER_RADIUS
                textColor = textColorString
                textFontSize = FONT_SIZE_TITLE
                textFontName = FONT_NAME
                leftMargin = margin
                rightMargin = margin
            }
            cancelButtonCustomization {
                backgroundColor = background
                cornerRadius = CORNER_RADIUS
                textColor = actionColor
                textFontSize = FONT_SIZE_16
                textFontName = FONT_NAME
            }
            resendButtonCustomization {
                backgroundColor = background
                cornerRadius = CORNER_RADIUS
                textColor = actionColor
                textFontSize = FONT_SIZE_TITLE
                textFontName = FONT_NAME
                leftMargin = margin
                rightMargin = margin
            }
        }
        return customization
    }

    private fun getSdkAppId(): UUID {
        val prefs = context.getSharedPreferences(PREFS_NAME, Context.MODE_PRIVATE)
        val sdkAppIdString = prefs.getString(SDK_APP_ID_KEY, null)
        var sdkAppId = try {
            sdkAppIdString?.let { UUID.fromString(it) }
        } catch (ignored: IllegalArgumentException) {
            null
        }
        val result = if (sdkAppId == null) {
            sdkAppId = UUID.randomUUID()
            prefs.edit().putString(SDK_APP_ID_KEY, sdkAppId.toString()).apply()
            sdkAppId
        } else {
            sdkAppId
        }

        return result
    }

    private fun List<ThreeDsCertInfo>.mapPsToDsId() = mutableMapOf<String, String>().apply {
        this@mapPsToDsId.forEach { this[it.paymentSystem] = it.dsId }
    }

    private fun getDsId(paymentSystem: String): String? = psToDsIdMap[paymentSystem]

    @Throws(Throwable::class)
    private fun initTransaction(
        threeDSWrapper: ThreeDSWrapper,
        threeDsVersion: String,
        paymentSystem: String,
        data: MutableMap<String, String>
    ): Transaction {
        val dsId = getDsId(paymentSystem)
        if (dsId == null) {
            threeDSWrapper.cleanupSafe(context)
            throw AcquiringSdkException(
                IllegalArgumentException(
                    "Directory server ID for payment system \"$paymentSystem\" can't be found"
                )
            )
        }
        var transaction: Transaction? = null
        try {
            transaction = threeDSWrapper.createTransaction(dsId, threeDsVersion)
            val authParams = transaction.authenticationRequestParameters

            data[SDK_MAP_APP_ID_KEY] = authParams.sdkAppID
            data[SDK_ENC_DATA_KEY] = android.util.Base64.encodeToString(
                authParams.deviceData.toByteArray(Charsets.UTF_8),
                ru.tinkoff.acquiring.sdk.utils.Base64.NO_WRAP
            )

            data[SDK_EPHEM_PUB_KEY] = android.util.Base64.encodeToString(
                authParams.sdkEphemeralPublicKey.toByteArray(Charsets.UTF_8),
                android.util.Base64.NO_WRAP
            )

            val display =
                (context.getSystemService(Context.WINDOW_SERVICE) as WindowManager).defaultDisplay
            val point = Point()
            display.getSize(point)

            data["timezone"] = getTimeZoneOffsetInMinutes()
            data["screen_height"] = "${point.y}"
            data["screen_width"] = "${point.x}"

            data[SDK_MAX_TIMEOUT_KEY] = "05"
            data[SDK_REFERENCE_NUMBER_KEY] = authParams.sdkReferenceNumber
            data[SDK_TRANS_ID_KEY] = authParams.sdkTransactionID
            data[SDK_INTERFACE_KEY] = "02"
            data[SDK_UI_TYPE_KEY] = "01,02,03,04,05"
        } catch (e: Throwable) {
            transaction?.closeSafe()
            threeDSWrapper.cleanupSafe(context)
            throw e
        }
        return transaction
    }

    private fun getTimeZoneOffsetInMinutes(): String {
        val offsetMills = TimeZone.getDefault().rawOffset
        return "${TimeUnit.MILLISECONDS.toMinutes(offsetMills.toLong())}"
    }
}

fun ThreeDSWrapper.cleanupSafe(context: Context) {
    if (isInitialized()) {
        try {
            cleanup(context)
        } catch (ignored: Throwable) {
            Log.e(TAG, ignored.message, ignored)
        }
    }
}