package ru.tinkoff.acquiring.sdk.threeds

import android.content.Context
import android.graphics.Point
import android.view.WindowManager
import android.webkit.WebView
import androidx.annotation.VisibleForTesting
import org.json.JSONObject
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.network.AcquiringApi
import ru.tinkoff.acquiring.sdk.network.AcquiringApi.COMPLETE_3DS_METHOD_V2
import ru.tinkoff.acquiring.sdk.responses.Check3dsVersionResponse
import ru.tinkoff.acquiring.sdk.utils.Base64
import ru.tinkoff.acquiring.sdk.utils.getTimeZoneOffsetInMinutes
import ru.tinkoff.acquiring.sdk.viewmodel.ThreeDsViewModel
import java.net.URLEncoder
import java.util.Locale

object ThreeDsHelper {

    object CollectData : ThreeDsDataCollector {

        private const val THREE_DS_CALLED_FLAG = "Y"
        private const val THREE_DS_NOT_CALLED_FLAG = "N"

        private val NOTIFICATION_URL
            get() = "${AcquiringApi.getUrl()}/$COMPLETE_3DS_METHOD_V2"
        private val TERM_URL_V2 = ThreeDsViewModel.TERM_URL_V2

        override operator fun invoke(context: Context, response: Check3dsVersionResponse?): MutableMap<String, String> {
            var threeDSCompInd = THREE_DS_NOT_CALLED_FLAG
            if (response?.threeDsMethodUrl != null) {
                val hiddenWebView = WebView(context)

                val threeDsMethodData = JSONObject().apply {
                    put("threeDSMethodNotificationURL", NOTIFICATION_URL)
                    put("threeDSServerTransID", response.serverTransId)
                }

                val dataBase64 = Base64.encodeToString(threeDsMethodData.toString().toByteArray(),
                    Base64.NO_PADDING or Base64.NO_WRAP).trim()
                val params = "threeDSMethodData=${URLEncoder.encode(dataBase64, "UTF-8")}"

                hiddenWebView.postUrl(response.threeDsMethodUrl!!, params.toByteArray())
                threeDSCompInd = THREE_DS_CALLED_FLAG
            }

            val display = (context.getSystemService(Context.WINDOW_SERVICE) as WindowManager).defaultDisplay
            val point = Point()
            display.getSize(point)

            return mutableMapOf<String, String>().apply {
                put("threeDSCompInd", threeDSCompInd)
                put("language", getLanguage(Locale.getDefault()))
                put("timezone", getTimeZoneOffsetInMinutes())
                put("screen_height", "${point.y}")
                put("screen_width", "${point.x}")
                put("cresCallbackUrl", TERM_URL_V2)
            }
        }

        @VisibleForTesting
        fun getLanguage(locale: Locale): String {
            return locale.language + "-" +  locale.country
        }

        fun addExtraData(data: MutableMap<String, String>, response: Check3dsVersionResponse) {
            data.apply {
                put("tdsServerTransId" , response.serverTransId!!)
            }
        }

        fun addExtraThreeDsData(data: ThreeDsData,
                                acsTransId: String,
                                serverTransId: String,
                                version:String
        ) {
            data.apply {
                this.version = version
                this.tdsServerTransId = serverTransId
                this.acsTransId = acsTransId
            }
        }
    }

    object Launch {
        const val RESULT_DATA = "result_data"
        const val ERROR_DATA = "result_error"
        const val RESULT_ERROR = 564
    }

    class CollectDataClass : ThreeDsDataCollector {

        private val THREE_DS_CALLED_FLAG = "Y"
        private val THREE_DS_NOT_CALLED_FLAG = "N"

        private val NOTIFICATION_URL
            get() = "${AcquiringApi.getUrl()}/$COMPLETE_3DS_METHOD_V2"
        private val TERM_URL_V2 = ThreeDsViewModel.TERM_URL_V2

        override operator fun invoke(context: Context, response: Check3dsVersionResponse?): MutableMap<String, String> {
            var threeDSCompInd = THREE_DS_NOT_CALLED_FLAG
            if (response?.threeDsMethodUrl != null) {
                val hiddenWebView = WebView(context)

                val threeDsMethodData = JSONObject().apply {
                    put("threeDSMethodNotificationURL", NOTIFICATION_URL)
                    put("threeDSServerTransID", response.serverTransId)
                }

                val dataBase64 = Base64.encodeToString(threeDsMethodData.toString().toByteArray(),
                    Base64.NO_PADDING or Base64.NO_WRAP).trim()
                val params = "threeDSMethodData=${URLEncoder.encode(dataBase64, "UTF-8")}"

                hiddenWebView.postUrl(response.threeDsMethodUrl!!, params.toByteArray())
                threeDSCompInd = THREE_DS_CALLED_FLAG
            }

            val display = (context.getSystemService(Context.WINDOW_SERVICE) as WindowManager).defaultDisplay
            val point = Point()
            display.getSize(point)

            return mutableMapOf<String, String>().apply {
                put("threeDSCompInd", threeDSCompInd)
                put("language", getLanguage(Locale.getDefault()))
                put("timezone", getTimeZoneOffsetInMinutes())
                put("screen_height", "${point.y}")
                put("screen_width", "${point.x}")
                put("cresCallbackUrl", TERM_URL_V2)
            }
        }

        @VisibleForTesting
        fun getLanguage(locale: Locale): String {
            return locale.language + "-" +  locale.country
        }

        fun addExtraData(data: MutableMap<String, String>, response: Check3dsVersionResponse) {
            data.apply {
                put("version", response.version!!)
                put("tdsServerTransId" , response.serverTransId!!)
            }
        }

        fun addExtraThreeDsData(data: ThreeDsData,
                                acsTransId: String,
                                serverTransId: String,
                                version:String
        ) {
            data.apply {
                this.version = version
                this.tdsServerTransId = serverTransId
                this.acsTransId = acsTransId
            }
        }
    }
}

interface ThreeDsDataCollector {

    operator fun invoke(context: Context, response: Check3dsVersionResponse?): MutableMap<String, String>
}
