package ru.tinkoff.acquiring.sdk.toggles

import android.util.Log
import com.google.gson.Gson
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.RequestBody.Companion.toRequestBody
import org.json.JSONObject
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine


/**
 * @author s.y.biryukov
 */
class FeatureToggleApi(
    private val client: OkHttpClient,
    private val gson: Gson
) {
    private fun createContextJson(sdkVersion: String, terminalKey: String?, appVersion: String?): JSONObject {
        return JSONObject()
            .put("asdk_version", sdkVersion)
            .put("platform", "Android")
            .put("terminalKey", terminalKey)
            .put("merch_app_version", appVersion)
    }

    private fun createUserIdsJson(deviceId: String?): JSONObject {
        return JSONObject()
            .put("deviceId", deviceId)
    }

    suspend fun loadFeatureToggles(
        terminalKey: String?,
        sdkVersion: String,
        deviceId: String?,
        appVersion: String?,
    ): List<FeatureToggleValue>? = suspendCoroutine {
        val jsonObject = JSONObject()
            .put("context", createContextJson(
                sdkVersion = sdkVersion,
                terminalKey = terminalKey,
                appVersion = appVersion
            ))
            .put("path", "/asdk/Android")
            .put("service", "asdk")
            .put("userIds", createUserIdsJson(deviceId))


        val jsonString = jsonObject.toString()

        val requestBody = jsonString.toRequestBody(("application/json").toMediaType())

        val request = Request.Builder()
            .url(getFeatureToggleEndpoint())
            .post(requestBody)
            .build()

        Log.d(TAG, "Request URL: ${request.url}")
        Log.d(TAG, "Request Body: $jsonObject")

        try {
            val response = client.newCall(request).execute()

            val result = response.use {
                if (response.isSuccessful) {
                    val featureToggleApiResponse =
                        gson.fromJson(
                            response.body?.charStream(),
                            FeatureToggleApiResponse::class.java
                        )
                    Log.d(TAG, "Response: $featureToggleApiResponse")
                    val items = featureToggleApiResponse.response.items ?: emptyList()
                    items.mapNotNull { item ->
                        when (item) {
                            is FeatureToggleApiResponse.Item.BooleanToggle -> FeatureToggleValue(
                                path = item.path,
                                booleanValue = item.value,
                                dcoStatusApplied = item.dcoAppliedStatus,
                            )

                            is FeatureToggleApiResponse.Item.StringToggle -> FeatureToggleValue(
                                path = item.path,
                                stringValue = item.value,
                                dcoStatusApplied = item.dcoAppliedStatus,
                            )

                            else -> null
                        }
                    }
                } else {
                    null
                }
            }
            it.resume(result)
        } catch (e: Exception) {
            Log.e(TAG, e.message, e)
            it.resume(null)
        }
    }

    private fun getFeatureToggleEndpoint(): String {
        return if (AcquiringSdk.isDeveloperMode) {
            "https://cfg-stage.dev-tcsgroup.io/api-gateway/v2/getToggles"
        } else {
            "https://cfg.tbank.ru/api-gateway/v2/getToggles"
        }
    }

    companion object {
        const val TAG = "TOGGLE"
    }
}
