package ru.tinkoff.acquiring.sdk.toggles

import com.google.gson.JsonDeserializationContext
import com.google.gson.JsonDeserializer
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.annotations.JsonAdapter
import com.google.gson.annotations.SerializedName
import java.lang.reflect.Type

/**
 * @author s.y.biryukov
 *
 * {"response":{"items":[{"value":false,"path":"asdk/Android/app_based_flow","type":"boolean","dco":{"dcoAppliedStatus":"no_placement"}}]}}
 */

data class FeatureToggleApiResponse(
    @SerializedName("response")
    val response: Response
) {
    data class Response(
        /**
         * Массив тоглов. Если по переданному пути не найдено ни одного тогла, вернем пустой массив
         */
        @SerializedName("items")
        val items: List<Item?>?
    )

    @JsonAdapter(FeatureItemDeserializer::class)
    sealed interface Item {
        /**
         * Идентификатор тогла
         */
        val path: String

        /**
         * Результат применения персонализации к тоглу.
         * applied - запрос попал в стратегию, value тогла перезаписано значением из вариации not_match - запрос не попал в стратегию, в value вернули дефолтное значение
         * no_placement - по запрашиваемому пути нет ни одной активной стратегии, в value вернули дефолтное значение
         * not_personalizable - персонализация тогла отключена в админке тоглов, в value вернули дефолтное значение
         * skipped - недостаточно данных для запроса персонализации (пустой service или userIds в запросе), в value вернули дефолтное значение
         * failure - ошибка применения персонализации (сетевые ошибки, не валидное значение в вариации и тд), в value вернули дефолтное значение
         */
        val dcoAppliedStatus: String

        data class BooleanToggle(
            val value: Boolean?,
            override val path: String,
            override val dcoAppliedStatus: String
        ) : Item

        data class StringToggle(
            val value: String?,
            override val path: String,
            override val dcoAppliedStatus: String
        ) : Item
    }
}

class FeatureItemDeserializer : JsonDeserializer<FeatureToggleApiResponse.Item?> {
    override fun deserialize(
        json: JsonElement,
        typeOfT: Type,
        context: JsonDeserializationContext
    ): FeatureToggleApiResponse.Item? {
        val jsonObject = json.asJsonObject
        val type = jsonObject.getAsJsonPrimitive("type").asString
        val toggle = when (type) {
            "boolean" -> parseBooleanToggle(jsonObject)
            "string" -> parseStringToggle(jsonObject)
            else -> null
        }
        return toggle
    }

    private fun parseBooleanToggle(jsonObject: JsonObject): FeatureToggleApiResponse.Item.BooleanToggle {
        return FeatureToggleApiResponse.Item.BooleanToggle(
            value = getBooleanValue(jsonObject),
            path = getPath(jsonObject),
            dcoAppliedStatus = getDcoAppliedStatus(jsonObject),
        )
    }

    private fun parseStringToggle(jsonObject: JsonObject): FeatureToggleApiResponse.Item.StringToggle {
        return FeatureToggleApiResponse.Item.StringToggle(
            value = getStringValue(jsonObject),
            path = getPath(jsonObject),
            dcoAppliedStatus = getDcoAppliedStatus(jsonObject),
        )
    }

    private fun getDcoAppliedStatus(jsonObject: JsonObject): String {
        return jsonObject.getAsJsonObject("dco")
            .getAsJsonPrimitive("dcoAppliedStatus").asString
    }

    private fun getPath(jsonObject: JsonObject): String =
        jsonObject.getAsJsonPrimitive("path").asString

    private fun getBooleanValue(jsonObject: JsonObject): Boolean =
        jsonObject.getAsJsonPrimitive("value").asBoolean

    private fun getStringValue(jsonObject: JsonObject): String =
        jsonObject.getAsJsonPrimitive("value").asString

}
