package ru.tinkoff.acquiring.sdk.toggles

import android.content.Context
import com.google.gson.Gson
import okhttp3.OkHttpClient
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.toggles.cache.InMemoryFeatureToggleCache
import ru.tinkoff.acquiring.sdk.toggles.cache.PersistenceFeatureToggleCache
import ru.tinkoff.acquiring.sdk.toggles.develop.OverrideStorage
import ru.tinkoff.acquiring.sdk.toggles.toggles.FeatureToggle

/**
 * @author s.y.biryukov
 */
interface FeatureToggleManager {
    /**
     * Загрузка удаленный фича-тоглов
     */
    suspend fun fetchToggles(): List<FeatureToggleValue>?

    /**
     * @return Список фича-тоглов
     */
    fun getAllToggles(): List<FeatureToggle>

    /**
     * @return значения для всех тогглов
     */
    suspend fun getAllTogglesValues(): Map<FeatureToggle, FeatureToggleValue>

    /**
     * @return включен ли заданный тоггл.
     */
    suspend fun isEnabled(featureToggle: FeatureToggle): Boolean

    /**
     * Переопределяет тогглы локально
     */
    suspend fun overrideValue(toggle: FeatureToggle, value: Boolean)

    /**
     * Удаляет тогглы переопределенные локально
     */
    suspend fun clearOverride()

    suspend fun updateFeatureToggleContext(modify:(FeatureToggleContext) -> FeatureToggleContext)

    fun getFeatureToggleContext(): FeatureToggleContext?

    fun setCacheTtl(seconds: Int)

    fun getCacheTtl(): Int

    /**
     * Очищает локальный кэш
     */
    suspend fun clearCache()

    suspend fun getFeatureToggleValue(featureToggle: FeatureToggle): FeatureToggleValue?

    data class FeatureToggleContext(
        val appVersion: String? = null,
        val terminalKey: String? = null,
        val deviceId: String? = null,
        val threeDsVersion: String = VERSION_NO
    ) {
        fun getKey(): String = "$terminalKey:$deviceId:$appVersion"

        companion object {
            const val VERSION_2_APP = "2.0 app"
            const val VERSION_2_BROWSER = "2.0 brw"
            const val VERSION_1 = "1.0"
            const val VERSION_NO = "NO"
        }
    }

    companion object {
        private var _instance: FeatureToggleManager? = null

        private fun init(
            context: Context
        ): FeatureToggleManager {
            val gson = Gson()
            val manager = FeatureToggleManagerImpl(
                featureToggleApi = FeatureToggleApi(
                    client = OkHttpClient(),
                    gson = gson,
                ),
                caches = listOf(
                    InMemoryFeatureToggleCache(),
                    PersistenceFeatureToggleCache(
                        context = context,
                        gson = gson,
                    )
                ),
                overrideStorage = if (AcquiringSdk.isDeveloperMode) {
                    OverrideStorage(
                        context = context,
                        gson = gson
                    )
                } else null
            )
            _instance = manager
            return manager
        }

        fun getInstance(context: Context): FeatureToggleManager = _instance ?: init(context)

        @Throws(NullPointerException::class)
        fun requireInstance(): FeatureToggleManager =
            requireNotNull(_instance) { "FeatureToggleManager не инициализирован" }
    }
}
