package ru.tinkoff.acquiring.sdk.toggles

import android.util.Log
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Deferred
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.async
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.joinAll
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.coroutines.withContext
import ru.tinkoff.acquiring.sdk.BuildConfig
import ru.tinkoff.acquiring.sdk.toggles.FeatureToggleManager.FeatureToggleContext
import ru.tinkoff.acquiring.sdk.toggles.cache.FeatureToggleCache
import ru.tinkoff.acquiring.sdk.toggles.develop.OverrideStorage
import ru.tinkoff.acquiring.sdk.toggles.toggles.FeatureToggle
import ru.tinkoff.acquiring.sdk.toggles.toggles.FeatureToggles
import java.util.Date

/**
 * @author s.y.biryukov
 */
class FeatureToggleManagerImpl(
    private val featureToggleApi: FeatureToggleApi,
    private val caches: List<FeatureToggleCache>,
    private val overrideStorage: OverrideStorage? = null
) : FeatureToggleManager {
    private val coroutineScope = CoroutineScope(Dispatchers.IO)
    private var cacheTtl: Int = DEFAULT_CACHE_TTL
    private var featureToggleContext: FeatureToggleContext? = null
    private var updateCacheJob: Job? = null
    private var fetchToggleDeferred: Deferred<List<FeatureToggleValue>?>? = null
    private val mutex: Mutex = Mutex()

    override suspend fun updateFeatureToggleContext(
        modify: (FeatureToggleContext) -> FeatureToggleContext
    ) {
        val currentContext = getFeatureToggleContext() ?: FeatureToggleContext()
        val updatedFeatureToggleContext = modify(currentContext)
        setFeatureToggleContext(updatedFeatureToggleContext)
    }

    override fun getFeatureToggleContext(): FeatureToggleContext? {
        return featureToggleContext
    }

    override fun setCacheTtl(seconds: Int) {
        cacheTtl = seconds
    }

    override fun getCacheTtl(): Int = cacheTtl

    override suspend fun isEnabled(featureToggle: FeatureToggle): Boolean = withContext(Dispatchers.IO) {
        val value = getFeatureToggleValue(featureToggle)?.booleanValue
        val defaultValue = featureToggle.defaultValue
        value ?: defaultValue
    }

    override suspend fun getFeatureToggleValue(featureToggle: FeatureToggle): FeatureToggleValue? {
        val value = getAllTogglesValues()[featureToggle]
        Log.d(TAG, "Чтение тоггла: ${featureToggle.path}=$value")
        return value
    }

    override suspend fun overrideValue(toggle: FeatureToggle, value: Boolean) {
        overrideStorage?.setFeatureToggleValue(toggle, value)
    }

    override suspend fun clearOverride() {
        overrideStorage?.clear()
    }

    override fun getAllToggles(): List<FeatureToggle> {
        return FeatureToggles.list
    }

    override suspend fun getAllTogglesValues(): Map<FeatureToggle, FeatureToggleValue> {
        val values = getToggleValues()
        val toggles = getAllToggles()
        return toggles.associateWith { toggle ->
            values.find { it.path == toggle.path } ?: FeatureToggleValue(
                path = toggle.path,
                booleanValue = toggle.defaultValue,
                source = Source.DEFAULT_LOCAL
            )
        }
    }

    private suspend fun getToggleValues(): List<FeatureToggleValue> {
        var toggles: List<FeatureToggleValue>? = getFromCache()
        if (toggles == null) {
            toggles = fetchToggles()
        }
        val togglesWithLocalOverride = applyOverrideIfNeed(toggles)
        return togglesWithLocalOverride
    }

    private fun applyOverrideIfNeed(values: List<FeatureToggleValue>?): List<FeatureToggleValue> {
        if (overrideStorage != null) {
            val valuesWithOverrides = values?.toMutableList() ?: mutableListOf()
            overrideStorage.getFeatureToggles()?.forEach { override ->
                val valueForOverride = valuesWithOverrides.find { it.path == override.path }

                if (valueForOverride != null) {
                    val prevValue = valueForOverride.booleanValue
                    val overrideValue = override.booleanValue
                    if(prevValue != overrideValue) {
                        Log.d(
                            TAG,
                            "Переопределен локально: ${valueForOverride.path} $prevValue -> $overrideValue"
                        )
                    }
                    valuesWithOverrides.remove(valueForOverride)
                }
                valuesWithOverrides.add(override)
            }
            return valuesWithOverrides
        }
        return values.orEmpty()
    }

    private suspend fun getFromCache(): List<FeatureToggleValue>? = withContext(Dispatchers.IO) {
        val cache = caches.find {
            !isCacheExpired(it)
        }
        val featureToggles = cache?.getFeatureToggles()
        if (cache != null && featureToggles != null) {
            val indexOfCache = caches.indexOf(cache)
            val cachesToUpdate = caches.take(indexOfCache)
            val createDate = cache.getCreateDate()
            updateCaches(cachesToUpdate, featureToggles, createDate!!)
        }

        featureToggles
    }

    private suspend fun isCacheExpired(cache: FeatureToggleCache): Boolean {
        val isContextChanged = cache.getFeatureToggleContextKey() != featureToggleContext?.getKey()
        val isObsolete = cache.getCreateDate() ?.let { getExpireDate(it).before(Date()) } ?: true
        return isContextChanged || isObsolete
    }

    override suspend fun fetchToggles(): List<FeatureToggleValue>? = withContext(Dispatchers.IO) {
        if (fetchToggleDeferred == null || fetchToggleDeferred?.isActive == false) {
            mutex.withLock {
                if (fetchToggleDeferred == null || fetchToggleDeferred?.isActive == false) {
                    fetchToggleDeferred = coroutineScope.async {
                        val featureToggleValues = loadFeatureToggleValues()
                        if (featureToggleValues?.isNotEmpty() == true) {
                            updateCaches(caches, featureToggleValues, Date())
                        }
                        featureToggleValues
                    }
                }
            }
        }
        fetchToggleDeferred?.await()
    }

    override suspend fun clearCache(): Unit = coroutineScope {
        caches.map {
            launch {
                it.clear()
            }
        }.joinAll()
    }

    private fun updateCaches(
        caches: List<FeatureToggleCache>,
        featureToggles: List<FeatureToggleValue>,
        createDate: Date
    ) {
        if (updateCacheJob != null) {
            updateCacheJob?.cancel()
        }
        updateCacheJob = coroutineScope.launch {
            caches.forEach {
                it.setFeatureToggles(featureToggles, createDate, featureToggleContext?.getKey())
            }
        }
    }

    private fun getExpireDate(date: Date) = Date(date.time + cacheTtl * MILLISECONDS_IN_SECOND)

    private suspend fun loadFeatureToggleValues(): List<FeatureToggleValue>? {
        val loadFeatureToggles = featureToggleApi.loadFeatureToggles(
            terminalKey = featureToggleContext?.terminalKey,
            sdkVersion = BuildConfig.ASDK_VERSION_NAME,
            deviceId = featureToggleContext?.deviceId,
            appVersion = featureToggleContext?.appVersion,
        )

        if (loadFeatureToggles.isNullOrEmpty()) {
            Log.d(TAG, "Не удалось загрузить удаленные feature toggle")
        }

        return loadFeatureToggles
    }

    private suspend fun setFeatureToggleContext(featureToggleContext: FeatureToggleContext?) {
        this.featureToggleContext = featureToggleContext

        if (caches.all { isCacheExpired(it) }) {
            coroutineScope.launch {
                fetchToggles()
            }
        }
    }

    companion object {
        const val DEFAULT_CACHE_TTL = 3600
        private const val TAG = "TOGGLE"
        private const val MILLISECONDS_IN_SECOND = 1000
    }
}
