package ru.tinkoff.acquiring.sdk.toggles.cache

import android.content.Context
import android.content.SharedPreferences
import androidx.core.content.edit
import com.google.gson.Gson
import com.google.gson.reflect.TypeToken
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import ru.tinkoff.acquiring.sdk.toggles.FeatureToggleValue
import ru.tinkoff.acquiring.sdk.toggles.Source
import java.util.Date

/**
 * @author s.y.biryukov
 */
open class PersistenceFeatureToggleCache(
    context: Context,
    private val gson: Gson
) : FeatureToggleCache {

    private val sharedPreferences: SharedPreferences =
        context.getSharedPreferences(SHARED_PREF_FEATURE_TOGGLE, Context.MODE_PRIVATE)

    override suspend fun getFeatureToggles(): List<FeatureToggleValue>? =
        withContext(Dispatchers.IO) {
            val json = sharedPreferences.getString(FEATURES_KEY, null) ?: return@withContext null
            val type =
                TypeToken.getParameterized(List::class.java, FeatureToggleValue::class.java).type
            gson.fromJson(json, type) as? List<FeatureToggleValue>?
        }

    override suspend fun setFeatureToggles(
        toggles: List<FeatureToggleValue>?,
        createDate: Date,
        contextKey: String?
    ) = withContext(Dispatchers.IO) {
        val fileToggles = toggles?.map { it.copy(source = Source.CACHE_DISK) }
        sharedPreferences.edit(commit = true) {
            putString(FEATURES_KEY, gson.toJson(fileToggles))
            putLong(CREATE_DATE_KEY, createDate.time)
            if (contextKey == null) {
                remove(CONTEXT_KEY)
            } else {
                putString(CONTEXT_KEY, contextKey)
            }
        }
    }

    override suspend fun getCreateDate(): Date? = withContext(Dispatchers.IO) {
        val createTime = sharedPreferences.getLong(CREATE_DATE_KEY, 0)
        if (createTime == 0L) null else Date(createTime)
    }

    override suspend fun clear(): Unit = withContext(Dispatchers.IO) {
        sharedPreferences.edit(commit = true) {
            remove(FEATURES_KEY)
            remove(CREATE_DATE_KEY)
            remove(CONTEXT_KEY)
        }
    }

    override suspend fun getFeatureToggleContextKey(): String? = withContext(Dispatchers.IO) {
        sharedPreferences.getString(CONTEXT_KEY, null)
    }

    companion object {
        private const val SHARED_PREF_FEATURE_TOGGLE = "toggle_cache"
        private const val FEATURES_KEY = "features_key"
        private const val CREATE_DATE_KEY = "create_date_key"
        private const val CONTEXT_KEY = "context_key"
    }
}
