package ru.tinkoff.acquiring.sdk.ui.activities

import android.os.Bundle
import android.os.Parcelable
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.core.os.bundleOf
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.navigation.fragment.NavHostFragment
import kotlinx.coroutines.launch
import kotlinx.parcelize.Parcelize
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.AcqQrCodeFlowFragmentBinding
import ru.tinkoff.acquiring.sdk.models.options.screen.BaseAcquiringOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.ui.fragments.DynamicQrCodeFragment
import ru.tinkoff.acquiring.sdk.ui.fragments.DynamicQrCodeFragmentDirections
import ru.tinkoff.acquiring.sdk.ui.fragments.StaticQrCodeFragment
import ru.tinkoff.acquiring.sdk.ui.fragments.StaticQrCodeFragmentDirections

internal class QrCodeFlowFragment : Fragment(), IsolatedKoinComponent {
    private var viewBinding: AcqQrCodeFlowFragmentBinding? = null

    private val viewModel by viewModel<QrCodeFlowViewModel> {
        parametersOf(
            SavedStateViewModelFactory(requireActivity().application, this, arguments)
        )
    }
    private var navHostFragment: NavHostFragment? = null

    override fun onResume() {
        super.onResume()
        viewModel.handleEvent(QrCodeFlowViewModel.Event.Resume)
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        val binding = AcqQrCodeFlowFragmentBinding.inflate(inflater, container, false)
        this.viewBinding = binding
        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        val fragment = childFragmentManager.findFragmentById(R.id.main_content) as NavHostFragment
        navHostFragment = fragment

        DynamicQrCodeFragment.registerResultListener(fragment.childFragmentManager, this) {
            when (it) {
                DynamicQrCodeFragment.Result.Cancel -> viewModel.handleEvent(QrCodeFlowViewModel.Event.PaymentCancel)
                is DynamicQrCodeFragment.Result.Failed -> viewModel.handleEvent(
                    QrCodeFlowViewModel.Event.PaymentFailed(
                        it.error
                    )
                )

                is DynamicQrCodeFragment.Result.Success -> viewModel.handleEvent(
                    QrCodeFlowViewModel.Event.PaymentSuccess(
                        it.paymentId
                    )
                )
            }
        }

        StaticQrCodeFragment.registerResultListener(fragment.childFragmentManager, this) {
            when (it) {
                StaticQrCodeFragment.Result.Cancel -> viewModel.handleEvent(QrCodeFlowViewModel.Event.PaymentCancel)
                is StaticQrCodeFragment.Result.Failed -> viewModel.handleEvent(
                    QrCodeFlowViewModel.Event.PaymentFailed(
                        it.error
                    )
                )

                is StaticQrCodeFragment.Result.Success -> viewModel.handleEvent(
                    QrCodeFlowViewModel.Event.PaymentSuccess(
                        it.paymentId
                    )
                )
            }
        }


        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.commandFlow.collect {
                    when (it) {
                        is QrCodeFlowViewModel.Command.OpenDynamicQr -> openDynamicQr(it.paymentOptions)
                        is QrCodeFlowViewModel.Command.OpenStaticQr -> openStaticQr(it.options)
                        QrCodeFlowViewModel.Command.FinishWithCancel -> returnResult(Result.Cancel)
                        is QrCodeFlowViewModel.Command.FinishWithError -> returnResult(
                            Result.Failed(it.error)
                        )

                        is QrCodeFlowViewModel.Command.FinishWithSuccess -> returnResult(
                            Result.Success(it.paymentId)
                        )
                    }
                }
            }
        }
    }

    override fun onDestroyView() {
        super.onDestroyView()
        navHostFragment = null
    }

    private fun openStaticQr(options: BaseAcquiringOptions) {
        val directions = StaticQrCodeFragmentDirections.openStaticQr(options)
        navHostFragment?.navController?.navigate(directions)
    }

    private fun openDynamicQr(paymentOptions: PaymentOptions) {
        val directions = DynamicQrCodeFragmentDirections.openDynamicQr(paymentOptions)
        navHostFragment?.navController?.navigate(directions)
    }

    private fun returnResult(result: Result) {
        parentFragmentManager.setFragmentResult(
            REQUEST_KEY, bundleOf(
                RESULT_KEY to result
            )
        )
    }

    companion object {
        private const val OPTIONS_ARG = "options"
        private const val TAG = "QrCodeFlowFragment"
        private const val REQUEST_KEY = "$TAG.REQUEST_KEY"
        private const val RESULT_KEY = "$TAG.RESULT_KEY"

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (Result) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(REQUEST_KEY, lifecycleOwner) { _, bundle ->
                bundle.getParcelable<Result>(RESULT_KEY)?.let(listener)
            }
        }

        fun newInstance(options: BaseAcquiringOptions): QrCodeFlowFragment {
            return QrCodeFlowFragment().apply {
                arguments = bundleOf(
                    OPTIONS_ARG to options
                )
            }
        }
    }

    sealed interface Result : Parcelable {
        @Parcelize
        data object Cancel : Result

        @Parcelize
        data class Success(
            val paymentId: Long? = null
        ) : Result

        @Parcelize
        data class Failed(
            val error: Throwable,
            val paymentId: Long? = null
        ) : Result
    }
}
