package ru.tinkoff.acquiring.sdk.ui.activities

import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.models.options.screen.BaseAcquiringOptions
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.utils.getExtra

/**
 * @author s.y.biryukov
 */
class QrCodeFlowViewModel(
    savedStateHandle: SavedStateHandle
) : ViewModel() {
    private var isResumeFirstTime: Boolean = true
    private val options: BaseAcquiringOptions = savedStateHandle.getExtra()

    private val _commandFlow = MutableSharedFlow<Command>()
    val commandFlow = _commandFlow.asSharedFlow()

    fun handleEvent(event: Event) {
        when (event) {
            Event.Resume -> onResume()
            Event.PaymentCancel -> onPaymentCancel()
            is Event.PaymentFailed -> onPaymentFailed(event.error)
            is Event.PaymentSuccess -> onPaymentSuccess(event.paymentId)
        }
    }

    private fun onPaymentCancel() {
        viewModelScope.launch {
            _commandFlow.emit(Command.FinishWithCancel)
        }
    }

    private fun onPaymentFailed(error: Throwable) {
        viewModelScope.launch {
            _commandFlow.emit(Command.FinishWithError(error))
        }
    }

    private fun onPaymentSuccess(paymentId: Long) {
        viewModelScope.launch {
            _commandFlow.emit(Command.FinishWithSuccess(paymentId))
        }
    }

    private fun onResume() {
        if (isResumeFirstTime) {
            isResumeFirstTime = false
            val command = when (options) {
                is PaymentOptions -> Command.OpenDynamicQr(options)
                else -> Command.OpenStaticQr(options)
            }
            viewModelScope.launch {
                _commandFlow.emit(command)
            }
        }
    }

    sealed interface Event {
        data object Resume : Event
        class PaymentSuccess(val paymentId: Long) : Event
        class PaymentFailed(val error: Throwable) : Event
        data object PaymentCancel : Event
    }

    sealed interface Command {
        class OpenStaticQr(
            val options: BaseAcquiringOptions
        ) : Command

        class OpenDynamicQr(
            val paymentOptions: PaymentOptions
        ) : Command

        data object FinishWithCancel : Command

        class FinishWithError(
            val error: Throwable
        ) : Command

        class FinishWithSuccess(
            val paymentId: Long
        ) : Command
    }
}
