/*
 * Copyright © 2020 Tinkoff Bank
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package ru.tinkoff.acquiring.sdk.ui.activities

import android.content.Context
import android.content.Intent
import android.os.Bundle
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.lifecycleScope
import kotlinx.coroutines.launch
import org.koin.android.ext.android.inject
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.models.PaymentSource
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.options.screen.BaseAcquiringOptions
import ru.tinkoff.acquiring.sdk.redesign.common.LauncherConstants
import ru.tinkoff.acquiring.sdk.redesign.common.LauncherConstants.EXTRA_CARD_PAN
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsHelper
import ru.tinkoff.acquiring.sdk.toggles.DeviceIdProvider
import ru.tinkoff.acquiring.sdk.toggles.FeatureToggleManager
import ru.tinkoff.acquiring.sdk.utils.getOptions

internal class ThreeDsActivity : AppCompatActivity(), IsolatedKoinComponent {

    private val featureToggleManager: FeatureToggleManager by inject()
    private val deviceIdProvider: DeviceIdProvider by inject()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.acq_activity_3ds)

        val options = intent.getOptions<BaseAcquiringOptions>()

        updateFeatureToggleContext(options)

        if (savedInstanceState == null) {
            val panSuffix = intent.getStringExtra(EXTRA_CARD_PAN) ?: ""
            val data = intent.getSerializableExtra(THREE_DS_DATA) as ThreeDsData
            val threeDsFragment = ThreeDsFragment.newInstance(
                threeDsData = data,
                panSuffix = panSuffix,
                options = options
            )
            supportFragmentManager.beginTransaction()
                .replace(R.id.acq_threeds_root, threeDsFragment)
                .commit()
        }

        setFragmentResultListener()
    }

    private fun updateFeatureToggleContext(options: BaseAcquiringOptions) {
        lifecycleScope.launch {
            featureToggleManager.updateFeatureToggleContext {
                it.copy(
                    terminalKey = options.terminalKey,
                    deviceId = deviceIdProvider.getDeviceId(),
                    appVersion = AcquiringSdk.appVersion,
                )
            }
        }
    }

    private fun setFragmentResultListener() {
        ThreeDsFragment.registerResultListener(supportFragmentManager, this) { result ->
            when (result) {
                is ThreeDsResult.Error -> {
                    val intent = Intent()
                    intent.putExtra(LauncherConstants.EXTRA_PAYMENT_ID, result.paymentId)
                    intent.putExtra(ThreeDsHelper.Launch.ERROR_DATA, result.error)
                    setResult(ThreeDsHelper.Launch.RESULT_ERROR, intent)
                    finish()
                }

                is ThreeDsResult.Success -> {
                    val intent = Intent()
                    intent.putExtra(ThreeDsHelper.Launch.RESULT_DATA, result.result)
                    setResult(RESULT_OK, intent)
                    finish()
                }

                ThreeDsResult.Cancel -> {
                    setResult(RESULT_CANCELED)
                    finish()
                }

                null -> Unit
            }
        }
    }



    companion object {

        private const val THREE_DS_DATA = "three_ds_data"
        private const val THREE_DS_PAY_SOURCE = "three_ds_pay_source"

        fun createIntent(
            context: Context,
            options: BaseAcquiringOptions,
            data: ThreeDsData,
            panSuffix: String = "",
            paymentSource: PaymentSource?
        ): Intent {
            val intent = Intent(context, ThreeDsActivity::class.java)
            intent.putExtra(THREE_DS_DATA, data)
            intent.putExtras(Bundle().apply {
                putParcelable(BaseAcquiringActivity.EXTRA_OPTIONS, options)
            })
            intent.putExtra(EXTRA_CARD_PAN, panSuffix)
            intent.putExtra(THREE_DS_PAY_SOURCE, paymentSource)
            return intent
        }
    }
}
