package ru.tinkoff.acquiring.sdk.ui.activities

import android.net.http.SslError
import android.os.Bundle
import android.util.Log
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.webkit.SslErrorHandler
import android.webkit.WebResourceError
import android.webkit.WebResourceRequest
import android.webkit.WebView
import android.webkit.WebViewClient
import androidx.activity.addCallback
import androidx.core.os.bundleOf
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import androidx.core.view.isInvisible
import androidx.core.view.isVisible
import androidx.core.view.updatePadding
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.databinding.FragmentThreeDsBinding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.options.screen.BaseAcquiringOptions
import ru.tinkoff.acquiring.sdk.models.result.AsdkResult
import ru.tinkoff.acquiring.sdk.ui.delegate.NotificationUtils
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData
import ru.tinkoff.acquiring.sdk.viewmodel.ThreeDsViewModel
import kotlin.math.max


internal class ThreeDsFragment : Fragment(), IsolatedKoinComponent {

    private var notificationBottomSheetBinding: NotificationUtils.NotificationBottomSheetBinding? =
        null

    private var viewBinding: FragmentThreeDsBinding? = null

    private val viewModel by viewModel<ThreeDsViewModel> {
        parametersOf(
            SavedStateViewModelFactory(requireActivity().application, this, arguments)
        )
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        addPaddingForKeyboard(view)

        requireActivity().onBackPressedDispatcher.addCallback(viewLifecycleOwner) {
            finishWithCancel()
        }

        viewBinding?.acq3dsWv?.run {
            isInvisible = true
            webViewClient = ThreeDsWebViewClient()
            settings.domStorageEnabled = true
            settings.javaScriptEnabled = true
            settings.javaScriptCanOpenWindowsAutomatically = true
        }

        this.notificationBottomSheetBinding =
            NotificationUtils.bindNotificationBottomSheet(activity = requireActivity())

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.stateFlow.collectLatest {

                    if (it.isWebViewInvisible) {
                        viewBinding?.acq3dsWv?.isVisible = false
                    }

                    val notification = it.notification
                    if (notification != null) {
                        val data = notification.toStatusViewData(requireContext())
                        notificationBottomSheetBinding?.show(data)
                    } else {
                        notificationBottomSheetBinding?.hide()
                    }
                }
            }
        }

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.commandFlow.collect {
                    when (it) {
                        is ThreeDsViewModel.Command.ReturnAddCardSuccessResult ->
                            finishWithSuccess(it.cardResult)

                        is ThreeDsViewModel.Command.ReturnError -> finishWithError(
                            throwable = it.error,
                            paymentId = it.paymentId
                        )

                        is ThreeDsViewModel.Command.ReturnPaymentSuccessResult ->
                            finishWithSuccess(it.paymentResult)

                        is ThreeDsViewModel.Command.StartThreeDs -> {
                            start3Ds(it.url, it.params)
                        }

                        ThreeDsViewModel.Command.ReturnCancel -> finishWithCancel()
                    }
                }
            }
        }
    }

    /**
     * Для того чтобы клавиатура не перекрывала
     */
    private fun addPaddingForKeyboard(view: View) {
        ViewCompat.setOnApplyWindowInsetsListener(view) { _, insets ->
            val imeBottom = insets.getInsets(WindowInsetsCompat.Type.ime()).bottom
            val navBottom = insets.getInsets(WindowInsetsCompat.Type.navigationBars()).bottom
            viewBinding?.root?.updatePadding(
                bottom = max(imeBottom, navBottom)
            )
            insets
        }
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        return FragmentThreeDsBinding.inflate(inflater, container, false)
            .also { this.viewBinding = it }
            .root
    }

    override fun onDestroyView() {
        super.onDestroyView()
        viewBinding?.acq3dsWv?.removeAllViews()
        viewBinding?.acq3dsWv?.destroy()
        this.viewBinding = null
    }

    override fun onResume() {
        super.onResume()
        viewModel.handleEvent(ThreeDsViewModel.Event.ViewResume)
    }

    private fun setSuccessResult(result: AsdkResult) {
        setResult(ThreeDsResult.Success(result))
    }

    private fun setErrorResult(throwable: Throwable, paymentId: Long?) {
        setResult(ThreeDsResult.Error(throwable, paymentId))
    }

    private fun setResult(result: ThreeDsResult) {
        parentFragmentManager.setFragmentResult(
            FRAGMENT_RESULT_KEY, bundleOf(
                FRAGMENT_RESULT_BUNDLE_KEY to result
            )
        )
    }

    private fun start3Ds(url: String, params: String) {
        viewBinding?.acq3dsWv?.postUrl(url, params.toByteArray())
    }

    private inner class ThreeDsWebViewClient : WebViewClient() {
        override fun onReceivedSslError(
            view: WebView,
            handler: SslErrorHandler,
            error: SslError
        ) {
            super.onReceivedSslError(view, handler, error)
            Log.e(TAG, "SSL error for URL ${error.url}")
        }
        override fun onReceivedError(
            view: WebView?,
            request: WebResourceRequest?,
            error: WebResourceError
        ) {
            super.onReceivedError(view, request, error)
            viewModel.handleEvent(
                ThreeDsViewModel.Event.WebPageLoadingError(
                    errorCode = error.errorCode,
                    description = error.description
                )
            )
        }

        override fun onPageFinished(view: WebView, url: String) {
            super.onPageFinished(view, url)
            view.isInvisible = false
            viewModel.handleEvent(ThreeDsViewModel.Event.WebPageLoaded(url))
        }
    }

    private fun finishWithSuccess(result: AsdkResult) {
        setSuccessResult(result)
    }

    private fun finishWithError(throwable: Throwable, paymentId: Long?) {
        setErrorResult(throwable, paymentId)
    }

    private fun finishWithCancel() {
        setResult(ThreeDsResult.Cancel)
    }

    companion object {
        private const val TAG = "ThreeDsFragment"
        private const val FRAGMENT_RESULT_KEY = "$TAG.FRAGMENT_RESULT_KEY"
        private const val FRAGMENT_RESULT_BUNDLE_KEY = "$TAG.FRAGMENT_RESULT_BUNDLE_KEY"

        fun newInstance(
            threeDsData: ThreeDsData,
            panSuffix: String,
            options: BaseAcquiringOptions
        ) =
            ThreeDsFragment().apply {
                arguments = ThreeDsFragmentArgs(
                    threeDsData = threeDsData,
                    panSuffix = panSuffix,
                    options = options,
                ).toBundle()
            }

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (ThreeDsResult) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(
                FRAGMENT_RESULT_KEY,
                lifecycleOwner
            ) { _, bundle ->
                val result = bundle.getParcelable<ThreeDsResult>(FRAGMENT_RESULT_BUNDLE_KEY)
                result?.let(listener)
            }
        }
    }
}
