package ru.tinkoff.acquiring.sdk.ui.delegate

import android.app.Activity
import android.app.ProgressDialog
import android.os.Parcelable
import android.util.Log
import androidx.lifecycle.LifecycleCoroutineScope
import com.emvco3ds.sdk.spec.CompletionEvent
import com.emvco3ds.sdk.spec.ProtocolErrorEvent
import com.emvco3ds.sdk.spec.RuntimeErrorEvent
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.parcelize.Parcelize
import ru.rtln.tds.sdk.transaction.ChallengeParameters
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringSdkException
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringSdkTimeoutException
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.enums.ResponseStatus
import ru.tinkoff.acquiring.sdk.models.result.PaymentResult
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsAppBasedTransaction
import ru.tinkoff.core.components.threedswrapper.ChallengeStatusReceiverAdapter
import ru.tinkoff.core.components.threedswrapper.ThreeDSWrapper.Companion.closeSafe
import java.lang.ref.WeakReference
import java.net.ProtocolException

private const val TIME_OUT = 5

private const val TAG = "AppBaseChallengeDelegate"

interface AppBaseChallengeDelegate {
    fun initAppBaseChallengeDelegate(
        activity: Activity,
        sdk: AcquiringSdk,
        lifecycleScope: LifecycleCoroutineScope
    )

    suspend fun launchChallenge(
        threeDsData: ThreeDsData,
        transaction: ThreeDsAppBasedTransaction,
        onResult: (AppBaseChallengeResult) -> Unit,
    )
}

class AppBaseChallengeDelegateImpl : AppBaseChallengeDelegate {

    private var activity: WeakReference<Activity> = WeakReference(null)
    private var scope: WeakReference<LifecycleCoroutineScope> = WeakReference(null)
    private var sdk: AcquiringSdk? = null

    override fun initAppBaseChallengeDelegate(
        activity: Activity,
        sdk: AcquiringSdk,
        lifecycleScope: LifecycleCoroutineScope,
    ) {
        this.activity = WeakReference(activity)
        this.scope = WeakReference(lifecycleScope)
        this.sdk = sdk
    }

    override suspend fun launchChallenge(
        threeDsData: ThreeDsData,
        transaction: ThreeDsAppBasedTransaction,
        onResult: (AppBaseChallengeResult) -> Unit
    ) {
        runCatching {
            val transaction3ds = transaction.transaction
            val challengeParams = ChallengeParameters().apply {
                acsTransactionID = (threeDsData.acsTransId)
                set3DSServerTransactionID(threeDsData.tdsServerTransId)
                acsRefNumber = threeDsData.acsRefNumber
                acsSignedContent = threeDsData.acsSignedContent
            }

            val progressDialog = try {
                transaction3ds.getProgressView(activity.get())
            } catch (e: Throwable) {
                transaction3ds.closeSafe()
                e.printStackTrace()
                Log.e(TAG, "Error: ${e::class.java.simpleName} ${e.message} ${e.printStackTrace()}")
                return
            }
            withContext(Dispatchers.IO) {
                transaction3ds.doChallenge(
                    activity.get(),
                    challengeParams,
                    object : ChallengeStatusReceiverAdapter(transaction3ds, progressDialog) {
                        override fun completed(event: CompletionEvent?) {

                            scope.get()?.launch {
                                submit3ds(
                                    challengeParams.get3DSServerTransactionID(),
                                    requireNotNull(event?.transactionStatus),
                                    requireNotNull(threeDsData.paymentId),
                                    onResult,
                                    requestKey = threeDsData.requestKey
                                )
                            }
                            closeChallenge(transaction3ds, progressDialog)
                        }

                        override fun cancelled() {
                            closeChallenge(transaction3ds, progressDialog)
                            onResult.invoke(AppBaseChallengeResult.Cancelled)
                        }

                        override fun timedout() {
                            closeChallenge(transaction3ds, progressDialog)
                            onResult.invoke(AppBaseChallengeResult.TimeOut(
                                AcquiringSdkTimeoutException(
                                    IllegalStateException("Challenge timeout")), requireNotNull(threeDsData.paymentId),
                                )
                            )
                        }

                        override fun protocolError(error: ProtocolErrorEvent?) {
                            closeChallenge(transaction3ds, progressDialog)
                            onResult.invoke(
                                AppBaseChallengeResult.ProtocolError(
                                    AcquiringSdkException(
                                        ProtocolException(error?.errorMessage?.errorDescription),
                                        paymentId = threeDsData.paymentId,
                                        errorCode = error?.errorMessage?.errorCode
                                    ),
                                    paymentId = requireNotNull(threeDsData.paymentId)
                                )
                            )
                        }

                        override fun runtimeError(error: RuntimeErrorEvent?) {
                            closeChallenge(transaction3ds, progressDialog)
                            onResult.invoke(
                                AppBaseChallengeResult.RuntimeError(
                                    AcquiringSdkException(
                                        RuntimeException(error?.errorMessage),
                                        paymentId = threeDsData.paymentId,
                                        errorCode = error?.errorCode
                                    ),
                                    paymentId = requireNotNull(threeDsData.paymentId)
                                )
                            )
                            Log.e(TAG, "$error")
                        }
                    },
                    TIME_OUT
                )
            }
        }.onFailure {
            onResult.invoke(AppBaseChallengeResult.Error(it, requireNotNull(threeDsData.paymentId)))
        }
    }

    private suspend fun submit3ds(
        threeDSServerTransID: String,
        transStatus: String,
        paymentId: Long,
        onResult: (AppBaseChallengeResult) -> Unit,
        requestKey: String?
    ) {
        scope.get()?.runCatching {
            onResult.invoke(AppBaseChallengeResult.Loading)
            val result = sdk?.submit3DSAuthorization(
                threeDSServerTransID = threeDSServerTransID,
                transStatus = transStatus
            )?.execute()
            val status = result?.status

            if (status == null) {
                onResult.invoke(
                    AppBaseChallengeResult.RuntimeError(
                        IllegalStateException("Payment status cannot be null"),
                        paymentId = paymentId,
                    )
                )
                return
            }

            when {
                ResponseStatus.checkUnhappyPassStatuses(status) -> {
                    onResult.invoke(
                        AppBaseChallengeResult.RuntimeError(
                            AcquiringSdkException(
                                IllegalStateException("Invalid response status"),
                                paymentId = paymentId
                            ),
                            paymentId = paymentId,
                        )
                    )
                }

                else -> {
                    onResult.invoke(
                        AppBaseChallengeResult.Success(
                            PaymentResult(
                                paymentId = requireNotNull(result.paymentId),
                            ),
                            requestKey = requestKey
                        )
                    )
                }
            }

        }?.onFailure {
            onResult.invoke(AppBaseChallengeResult.Error(it, paymentId = paymentId))
        }
    }

    private fun closeChallenge(
        transaction3ds: com.emvco3ds.sdk.spec.Transaction,
        progressDialog: ProgressDialog
    ) {
        progressDialog.dismiss()
    }
}

sealed interface AppBaseChallengeResult : Parcelable {

    @Parcelize
    data class Success(
        val paymentResult: PaymentResult,
        val requestKey: String? = null
    ) : AppBaseChallengeResult

    sealed interface AppBaseChallengeResultError: AppBaseChallengeResult {
        val error: Throwable
        val paymentId: Long
    }

    @Parcelize
    data class Error(
        override val error: Throwable,
        override val paymentId: Long
    ) : AppBaseChallengeResultError

    @Parcelize
    data class RuntimeError(
        override val error: Throwable,
        override val paymentId: Long
    ) : AppBaseChallengeResultError

    @Parcelize
    data class ProtocolError(
        override val error: Throwable,
        override val paymentId: Long
    ) : AppBaseChallengeResultError

    @Parcelize
    data class TimeOut(
        override val error: Throwable,
        override val paymentId: Long,
    ) : AppBaseChallengeResultError

    @Parcelize
    data object Cancelled : AppBaseChallengeResult

    @Parcelize
    data object Loading : AppBaseChallengeResult
}
