package ru.tinkoff.acquiring.sdk.ui.delegate

import ru.tinkoff.acquiring.sdk.redesign.common.carddatainput.CardNumberFormatter
import ru.tinkoff.acquiring.sdk.ui.customview.editcard.CardPaymentSystem
import ru.tinkoff.acquiring.sdk.ui.customview.editcard.validators.CardValidator

/**
 * @author s.y.biryukov
 */
interface CardDataInputDelegate {
    fun validateCardNumberAndGoToNextField(
        cardNumber: String,
        allowGotoNextField: Boolean,
        forceHideError: Boolean,
        isFocused: Boolean,
        updateState: (cardNumber: String, isValidCardNumber: Boolean?) -> Unit,
        goToNextField: () -> Unit,
    )

    fun validateExpireDateAndFocusNextField(
        expireDate: String,
        allowGoToNextField: Boolean,
        isFocused: Boolean,
        updateState: (expireDate: String, isValidExpireDate: Boolean?) -> Unit,
        goToNextField: () -> Unit
    )

    fun validateCvc(
        cvc: String,
        isFocused: Boolean,
        updateState: (cvc: String, isCvcValid: Boolean?) -> Unit
    )

    fun onCardScanned(
        cardNumber: String?,
        expireDate: String?,
        updateState: (cardNumber: String?, expireDate: String?) -> Unit,
        requestExpireDateFocus: () -> Unit,
        requestCvcFocus: () -> Unit
    )
}

class CardDataInputDelegateImpl: CardDataInputDelegate {
    override fun validateCardNumberAndGoToNextField(
        cardNumber: String,
        allowGotoNextField: Boolean,
        forceHideError: Boolean,
        isFocused: Boolean,
        updateState: (cardNumber: String, isValidCardNumber: Boolean?) -> Unit,
        goToNextField: () -> Unit
    ) {
        val normalizedCardNumber = CardNumberFormatter.normalize(cardNumber)
        var isValid: Boolean? = CardValidator.validateCardNumber(normalizedCardNumber)

        if (isValid == false && (!shouldShowCardNumberError(normalizedCardNumber, isFocused) || forceHideError)) {
            isValid = null
        }

        updateState(normalizedCardNumber, isValid)

        if (allowGotoNextField && CardValidator.shouldAutoSwitchFromCardNumber(normalizedCardNumber)) {
            goToNextField()
        }
    }

    override fun validateExpireDateAndFocusNextField(
        expireDate: String,
        allowGoToNextField: Boolean,
        isFocused: Boolean,
        updateState: (cardNumber: String, isValidCardNumber: Boolean?) -> Unit,
        goToNextField: () -> Unit
    ) {
        var isValid: Boolean? = CardValidator.validateExpireDate(expireDate, false)
        if (isValid == false && !shouldShowExpireDateError(isFocused)) {
            isValid = null
        }

        updateState(expireDate, isValid)

        if (allowGoToNextField && shouldGoToNextFieldFromExpireDate(expireDate)) {
            goToNextField()
        }
    }

    override fun validateCvc(
        cvc: String,
        isFocused: Boolean,
        updateState: (cvc: String, isCvcValid: Boolean?) -> Unit
    ) {
        var isValid: Boolean? = CardValidator.validateSecurityCode(cvc)
        if (isValid == false && !shouldShowCvcError(cvc, isFocused)) {
            isValid = null
        }
        updateState(cvc, isValid)
    }

    override fun onCardScanned(
        cardNumber: String?,
        expireDate: String?,
        updateState: (cardNumber: String?, expireDate: String?) -> Unit,
        requestExpireDateFocus: () -> Unit,
        requestCvcFocus: () -> Unit,
    ) {
        updateState(cardNumber, expireDate)

        if (expireDate.isNullOrBlank()) {
            requestExpireDateFocus()
        } else {
            requestCvcFocus()
        }
    }

    private fun shouldShowCardNumberError(normalizedCardNumber: String, isFocused: Boolean): Boolean {
        if (!isFocused) return true
        val paySystem = CardPaymentSystem.resolve(normalizedCardNumber)
        val length = normalizedCardNumber.length
        return paySystem == CardPaymentSystem.UNKNOWN && length >= 6
    }

    private fun shouldGoToNextFieldFromExpireDate(expireDate: String): Boolean {
        return expireDate.length == CardValidator.MAX_DATE_LENGTH
    }

    private fun shouldShowCvcError(cvc: String, isFocused: Boolean): Boolean {
        return !isFocused
    }

    private fun shouldShowExpireDateError(isFocused: Boolean): Boolean {
        return !isFocused
    }
}
