package ru.tinkoff.acquiring.sdk.ui.delegate

import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.activity.ComponentActivity
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import androidx.core.view.isVisible
import androidx.core.view.updatePadding
import androidx.lifecycle.lifecycleScope
import com.google.android.material.bottomsheet.BottomSheetBehavior
import com.google.android.material.bottomsheet.BottomSheetBehavior.BottomSheetCallback
import com.google.android.material.bottomsheet.BottomSheetBehavior.STATE_COLLAPSED
import com.google.android.material.bottomsheet.BottomSheetBehavior.STATE_EXPANDED
import com.google.android.material.bottomsheet.BottomSheetBehavior.STATE_HIDDEN
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.AcqStatusBottomSheetBinding
import ru.tinkoff.acquiring.sdk.ui.customview.status.MyBottomSheetBehaviour
import ru.tinkoff.acquiring.sdk.ui.customview.status.StatusViewData

/**
 * @author s.y.biryukov
 */
class NotificationUtils {
    companion object {
        fun bindNotificationBottomSheet(
            activity: ComponentActivity
        ): NotificationBottomSheetBinding {
            val container = activity.window.decorView.findViewById<ViewGroup>(android.R.id.content)


            val dataFlow = MutableSharedFlow<StatusViewData?>()

            val notificationBinding = object : NotificationBottomSheetBinding() {
                override fun show(data: StatusViewData) {
                    closeReason = CloseReason.USER
                    activity.lifecycleScope.launch {
                        dataFlow.emit(data)
                    }
                }

                override fun hide(closeReason: CloseReason) {
                    this.closeReason = closeReason
                    activity.lifecycleScope.launch {
                        dataFlow.emit(null)
                    }
                }
            }

            activity.lifecycleScope.launch {
                dataFlow.collectLatest { data ->
                    if (data != null) {
                        val binding = requireNotNull(getNotificationViewBinding(container, true)) {
                            "Не удалось инициализировать шторку состояний"
                        }

                        val behavior = binding.let {
                            (BottomSheetBehavior.from(binding.acqBottomSheet) as MyBottomSheetBehaviour)
                        }.apply {
                            skipCollapsed = true
                        }
                        if (behavior.state == STATE_EXPANDED) {
                            delay(2000)
                        } else {
                            behavior.state = STATE_EXPANDED
                            binding.root.isVisible = true
                        }
                        binding.acqStatusView.showNotification(data)
                        val newBottomSheetCallback = object : BottomSheetCallback() {
                            override fun onStateChanged(view: View, stage: Int) {
                                if (stage == STATE_HIDDEN) {
                                    behavior.removeBottomSheetCallback(this)
                                    binding.root.isVisible = false
                                    (binding.root.parent as ViewGroup).removeView(binding.root)
                                    if (notificationBinding.closeReason == CloseReason.USER) {
                                        val swipeAction = data.onCloseByUser ?: data.onButtonClick
                                        swipeAction?.invoke()
                                    }
                                }
                            }

                            override fun onSlide(view: View, fraction: Float) = Unit
                        }
                        behavior.setBottomSheetCallback(newBottomSheetCallback)
                    } else {
                        val binding = getNotificationViewBinding(container, false)
                        binding?.let {
                            val behaviour =
                                (BottomSheetBehavior.from(binding.acqBottomSheet) as? MyBottomSheetBehaviour)
                            behaviour?.setBottomSheetCallback(null)
                            binding.root.isVisible = false
                            (binding.root.parent as ViewGroup).removeView(binding.root)
                            behaviour?.state = STATE_HIDDEN
                        }
                    }
                }
            }

            return notificationBinding
        }

        private fun getNotificationViewBinding(
            container: ViewGroup,
            createIfNotExist: Boolean
        ): AcqStatusBottomSheetBinding? {
            val viewId = R.id.acq_error_bottom_sheet
            val attachedBottomSheetView = container.findViewById<View>(viewId)
            var binding: AcqStatusBottomSheetBinding? = null
            if (attachedBottomSheetView != null) {
                binding = AcqStatusBottomSheetBinding.bind(attachedBottomSheetView)
            } else if (createIfNotExist) {
                val context = container.context
                binding = AcqStatusBottomSheetBinding.inflate(
                    LayoutInflater.from(context),
                    container,
                    true
                )
                binding.root.id = viewId
                val behavior =
                    (BottomSheetBehavior.from(binding.acqBottomSheet) as MyBottomSheetBehaviour)
                behavior.state = STATE_COLLAPSED

                ViewCompat.setOnApplyWindowInsetsListener(container) {
                        view, insets ->
                    binding.acqBottomSheet.updatePadding(
                        bottom = insets.getInsets(WindowInsetsCompat.Type.navigationBars()).bottom
                    )
                    insets
                }
            }
            return binding
        }
    }

    abstract class NotificationBottomSheetBinding {
        internal var closeReason: CloseReason = CloseReason.USER
        abstract fun show(data: StatusViewData)
        abstract fun hide(closeReason: CloseReason = CloseReason.CODE)
    }

    enum class CloseReason {
        CODE, USER
    }
}
