package ru.tinkoff.acquiring.sdk.ui.fragments

import android.os.Bundle
import android.os.Parcelable
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.core.os.bundleOf
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.setFragmentResult
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.navigation.fragment.NavHostFragment
import androidx.navigation.navOptions
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import kotlinx.parcelize.Parcelize
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.AttachCardGraphDirections
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.AcqAttachCardFlowFragmentBinding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.options.screen.AttachCardOptions
import ru.tinkoff.acquiring.sdk.ui.activities.ThreeDsFragment
import ru.tinkoff.acquiring.sdk.ui.activities.ThreeDsResult
import ru.tinkoff.acquiring.sdk.ui.delegate.NotificationUtils
import ru.tinkoff.acquiring.sdk.ui.fragments.v2.AttachCardFragmentV2
import ru.tinkoff.acquiring.sdk.utils.AcqSnackBarHelper
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData
import ru.tinkoff.acquiring.sdk.viewmodel.AttachCardFlowEvent
import ru.tinkoff.acquiring.sdk.viewmodel.AttachCardFlowViewModel

/**
 * @author s.y.biryukov
 */
class AttachCardFlowFragment : Fragment(), IsolatedKoinComponent {
    private var viewBinding: AcqAttachCardFlowFragmentBinding? = null
    private var navHostFragment: NavHostFragment? = null
    private val viewModel: AttachCardFlowViewModel by viewModel {
        parametersOf(
            SavedStateViewModelFactory(
                requireActivity().application, this, arguments
            )
        )
    }

    private var notificationBottomSheetBinding: NotificationUtils.NotificationBottomSheetBinding? =
        null
    private var snackBarHelper: AcqSnackBarHelper? = null

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        val binding = AcqAttachCardFlowFragmentBinding.inflate(inflater, container, false)
        viewBinding = binding
        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        this.snackBarHelper = AcqSnackBarHelper(view)

        val fragment = childFragmentManager.findFragmentById(R.id.main_content) as NavHostFragment
        navHostFragment = fragment

        this.notificationBottomSheetBinding = NotificationUtils.bindNotificationBottomSheet(
            activity = requireActivity()
        )

        AttachCardFragment.registerResultListener(fragment.childFragmentManager, this) {
            handleAttachCardResult(it)
        }

        AttachCardFragmentV2.registerResultListener(fragment.childFragmentManager, this) {
            handleAttachCardResult(it)
        }

        ThreeDsFragment.registerResultListener(fragment.childFragmentManager, this) {
            handleThreeDsResult(it)
        }


        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.stateFlow.collectLatest {
                    val notification = it.notification
                    if (notification == null) {
                        notificationBottomSheetBinding?.hide()
                    } else {
                        val data = notification.toStatusViewData(requireContext())
                        notificationBottomSheetBinding?.show(data)
                    }
                }
            }
        }

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {

                viewModel.commandFlow.collect {
                    when (it) {
                        is AttachCardFlowViewModel.Command.OpenAttachCardScreen -> {
                            showAttachCardFragment(
                                options = it.options,
                                showError = it.showError,
                                isNewCardViewToggleEnabled = it.isNewCardViewToggleEnabled
                            )
                        }

                        is AttachCardFlowViewModel.Command.FinishWithAttachedCard -> {
                            returnResult(
                                Result.Success(
                                    cardId = it.cardId,
                                    panSuffix = it.panSuffix
                                )
                            )
                        }

                        is AttachCardFlowViewModel.Command.RequireThreeDs -> {
                            requireThreeDs(
                                threeDsData = it.threeDsData,
                                panSuffix = it.panSuffix,
                                options = it.attachCardOptions,
                            )
                        }

                        is AttachCardFlowViewModel.Command.FinishWithError -> {
                            returnResult(
                                Result.Failed(
                                    error = it.error,
                                    panSuffix = it.panSuffix
                                )
                            )
                        }

                        is AttachCardFlowViewModel.Command.ShowError -> {
                            showErrorToast(
                                message = it.message,
                                args = it.args
                            )
                        }

                        AttachCardFlowViewModel.Command.FinishWithCancel -> {
                            returnResult(Result.Cancel)
                        }

                        AttachCardFlowViewModel.Command.ReturnToAttachCardScreen -> {
                            navHostFragment?.navController?.popBackStack(
                                destinationId = R.id.attachCardFragmentV2,
                                inclusive = false
                            )
                        }
                    }
                }
            }
        }
    }

    override fun onResume() {
        super.onResume()
        viewModel.handleEvent(AttachCardFlowEvent.Resume)
    }


    private fun showErrorToast(args: List<Any>, message: Int) {
        snackBarHelper?.showWithIcon(
            R.drawable.acq_ic_alert_ngative,
            if (args.isNotEmpty()) {
                getString(message, *args.toTypedArray())
            } else {
                getString(message)
            }
        )
    }

    private fun handleThreeDsResult(it: ThreeDsResult?) {
        when (it) {
            ThreeDsResult.Cancel -> viewModel.handleEvent(AttachCardFlowEvent.ThreeDsCancelled)
            is ThreeDsResult.Error -> viewModel.handleEvent(
                AttachCardFlowEvent.ThreeDsError(
                    error = it.error,
                    paymentId = it.paymentId
                )
            )

            is ThreeDsResult.Success -> viewModel.handleEvent(
                AttachCardFlowEvent.ThreeDsSuccess(
                    it.result
                )
            )

            null -> Unit
        }
    }

    override fun onDestroyView() {
        viewBinding = null
        snackBarHelper = null
        super.onDestroyView()
    }

    private fun handleAttachCardResult(result: AttachCardFragment.Result?) {
        when (result) {
            is AttachCardFragment.Result.Error -> viewModel.handleEvent(
                AttachCardFlowEvent.CardAttachError(
                    error = result.error,
                    panSuffix = result.panSuffix,
                )
            )

            is AttachCardFragment.Result.CardAdded -> viewModel.handleEvent(
                AttachCardFlowEvent.CardAttached(result.card)
            )

            is AttachCardFragment.Result.Cancelled -> viewModel.handleEvent(
                AttachCardFlowEvent.CardAttachCancelled
            )

            is AttachCardFragment.Result.ThreeDsRequired -> viewModel.handleEvent(
                AttachCardFlowEvent.RequireThreeDs(
                    data = result.data,
                    panSuffix = result.panSuffix
                )
            )
            is AttachCardFragment.Result.TimeOut -> viewModel.handleEvent(
                AttachCardFlowEvent.TimeOut(
                    error = result.error,
                    panSuffix = result.panSuffix
                )
            )
            is AttachCardFragment.Result.ChallengeResult -> viewModel.handleEvent(
                AttachCardFlowEvent.ChallengeResult(
                    data = result.result,
                    card = result.card
                )
            )

            null -> Unit
        }
    }

    private fun requireThreeDs(
        threeDsData: ThreeDsData,
        panSuffix: String,
        options: AttachCardOptions
    ) {
        navHostFragment?.navController?.navigate(
            AttachCardGraphDirections.openThreeDs(
                threeDsData = threeDsData,
                panSuffix = panSuffix,
                options = options
            )
        )
    }

    private fun showAttachCardFragment(options: AttachCardOptions, showError: Boolean, isNewCardViewToggleEnabled: Boolean) {
        if (isNewCardViewToggleEnabled) {
            navHostFragment?.navController?.navigate(
                AttachCardGraphDirections.openAttachCardV2(
                    attachCardOptions = options,
                ),
                navOptions = navOptions { launchSingleTop = true }
            )
        } else {
            navHostFragment?.navController?.navigate(
                AttachCardGraphDirections.openAttachCard(
                    attachCardOptions = options,
                    showError = showError
                ),
                navOptions = navOptions { launchSingleTop = true }
            )
        }
    }

    private fun returnResult(result: Result) {
        setFragmentResult(
            RESULT_KEY, bundleOf(
                RESULT_DATA_KEY to result
            )
        )
    }

    companion object {
        private const val RESULT_KEY = "AttachCardFlowFragment.RESULT_KEY"
        private const val RESULT_DATA_KEY = "AttachCardFlowFragment.RESULT_DATA_KEY"

        fun newInstance(
            options: AttachCardOptions,
            showError: Boolean,
            isNewCardViewToggleEnabled: Boolean
        ): AttachCardFlowFragment {
            return AttachCardFlowFragment().apply {
                arguments = AttachCardFlowFragmentArgs(
                    attachCardOptions = options,
                    showError = showError,
                ).toBundle()
            }
        }

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (Result) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(RESULT_KEY, lifecycleOwner) { _, bundle ->
                val result = bundle.getParcelable<Result>(RESULT_DATA_KEY)
                result?.let(listener)
            }
        }
    }

    sealed interface Result : Parcelable {
        @Parcelize
        data class Success(
            val cardId: String,
            val panSuffix: String
        ) : Result

        @Parcelize
        data class Failed(
            val panSuffix: String?,
            val error: Throwable
        ) : Result

        @Parcelize
        data object Cancel : Result
    }
}
