/*
 * Copyright © 2020 Tinkoff Bank
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package ru.tinkoff.acquiring.sdk.ui.fragments

import android.os.Bundle
import android.os.Parcelable
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.activity.addCallback
import androidx.core.os.bundleOf
import androidx.core.view.isVisible
import androidx.core.view.postDelayed
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.commit
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import kotlinx.coroutines.launch
import kotlinx.parcelize.Parcelize
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.AcqFragmentAttachCardBinding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringSdkException
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.options.screen.AttachCardOptions
import ru.tinkoff.acquiring.sdk.models.paysources.CardData
import ru.tinkoff.acquiring.sdk.models.result.CardResult
import ru.tinkoff.acquiring.sdk.redesign.common.carddatainput.CardDataInputFragment
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.smartfield.AcqTextFieldView
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeDelegate
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeDelegateImpl
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeResult
import ru.tinkoff.acquiring.sdk.utils.KeyboardVisionUtils
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData
import ru.tinkoff.acquiring.sdk.viewmodel.AttachCardViewModel

internal class AttachCardFragment : Fragment(),
    CardDataInputFragment.OnCardDataChanged,
    AppBaseChallengeDelegate by AppBaseChallengeDelegateImpl(), IsolatedKoinComponent {

    private var cardDataInputFragment: CardDataInputFragment? = null
    private val viewModel by viewModel<AttachCardViewModel> {
        parametersOf(
            SavedStateViewModelFactory(requireActivity().application, this, arguments)
        )
    }

    private var viewBinding: AcqFragmentAttachCardBinding? = null

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        val binding = AcqFragmentAttachCardBinding.inflate(inflater, container, false)
        this.viewBinding = binding
        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        requireActivity().onBackPressedDispatcher.addCallback(viewLifecycleOwner) {
            viewModel.handleEvent(AttachCardViewModel.Event.BackButtonClicked)
        }

        initToolbar()

        viewBinding?.acqAttachBtnAttach?.setOnClickListener {
            hideKeyboard()
            viewModel.handleEvent(AttachCardViewModel.Event.AttachButtonClicked)
        }

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.stateFlow.collect {
                    updateCardDataInputFragment(it.options)

                    handleLoadState(it.isLoading)

                    handleNotification(it.notification)
                }
            }
        }

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.commandFlow.collect {
                    when (it) {
                        is AttachCardViewModel.Command.FinishWithSuccess -> sendResult(
                            Result.CardAdded(it.cardResult)
                        )

                        is AttachCardViewModel.Command.FinishWithError -> sendResult(
                            Result.Error(
                                error = it.error,
                                errorCode = (it.error as? AcquiringSdkException)?.errorCode?.toIntOrNull(),
                                panSuffix = it.panSuffix,
                            )
                        )

                        is AttachCardViewModel.Command.Open3DSScreen -> sendResult(
                            Result.ThreeDsRequired(
                                data = it.data,
                                panSuffix = it.panSuffix,
                                options = it.options
                            )
                        )

                        is AttachCardViewModel.Command.FinishWithTimeOut -> {
                            sendResult(Result.TimeOut(
                                error = it.error,
                                panSuffix = it.panSuffix
                            ))
                        }

                        is AttachCardViewModel.Command.Start3DsChallenge -> {
                            initAppBaseChallengeDelegate(
                                requireActivity(),
                                AcquiringSdk(
                                    it.options.terminalKey,
                                    it.options.publicKey,
                                ),
                                viewLifecycleOwner.lifecycleScope
                            )
                            launchChallenge(
                                it.threeDsData,
                                it.transaction,
                                onResult = { result: AppBaseChallengeResult ->
                                    processChallengeResult(result, it.cardResult)
                                },
                            )
                        }

                        is AttachCardViewModel.Command.Cancel -> {
                            sendResult(Result.Cancelled)
                        }

                        AttachCardViewModel.Command.RequestCardFieldFocus -> {
                            setFocusToCardNumber()
                        }
                    }
                }
            }
        }
    }

    private fun setFocusToCardNumber() {
        requireView().postDelayed(500) { // TODO Убрать после переработки CardDataInputFragment
            val cardDataInput = cardDataInputFragment
            if (cardDataInput != null && cardDataInput.isAdded && cardDataInput.isVisible) {
                cardDataInput.requestFocusOnCardNumber()
            }
        }
    }

    override fun onResume() {
        super.onResume()
        viewModel.handleEvent(AttachCardViewModel.Event.Resume)
    }

    private fun updateCardDataInputFragment(attachCardOptions: AttachCardOptions) {
        var cardDataInputFragment: CardDataInputFragment? = this.cardDataInputFragment

        if(cardDataInputFragment == null) {
            cardDataInputFragment = childFragmentManager
                .findFragmentById(R.id.fragment_card_data_input) as? CardDataInputFragment
        }
        if (cardDataInputFragment == null) {
            cardDataInputFragment = CardDataInputFragment.newInstance(attachCardOptions)
            childFragmentManager.commit {
                add(R.id.fragment_card_data_input, cardDataInputFragment)
                addToBackStack(null)
            }
        }
        cardDataInputFragment.onNext = {
            if (cardDataInputFragment.isCvcInput(it)) {
                (it as? AcqTextFieldView)?.hideKeyboard()
                true
            } else false
        }
        this.cardDataInputFragment = cardDataInputFragment
    }

    private fun hideKeyboard() {
        requireActivity().currentFocus?.clearFocus()
        KeyboardVisionUtils.hideKeyboard(requireView())
    }

    override fun onDestroyView() {
        super.onDestroyView()
        cardDataInputFragment = null
        viewBinding = null
    }

    private fun initToolbar() {
        viewBinding?.acqToolbar?.apply {
            setTitle(R.string.acq_cardlist_addcard_title)
            setNavigationOnClickListener {
                viewModel.handleEvent(AttachCardViewModel.Event.BackButtonClicked)
            }
        }
    }

    private fun handleNotification(notification: Notification?) {
        if (notification == null) {
            viewBinding?.acqAttachCardStatusViewWrapper?.isVisible = false
        } else {
            val data = notification.toStatusViewData(requireContext())
            viewBinding?.acqAttachCardStatusViewWrapper?.isVisible = true
            viewBinding?.acqAttachCardStatusView?.showNotification(data)
        }
    }

    override fun onCardDataChanged(
        data: CardDataInputFragment.OnCardDataChanged.Data,
        isValid: Boolean
    ) {
        viewBinding?.acqAttachBtnAttach?.isEnabled = isValid
        viewModel.handleEvent(
            AttachCardViewModel.Event.CardDataChanged(
                cardData = CardData(
                    data.pan,
                    data.expiryDate,
                    data.securityCode
                ),
                isValid = isValid
            )
        )
    }

    private fun handleLoadState(isLoading: Boolean) {
        viewBinding?.acqAttachBtnAttach?.isLoading = isLoading
        viewBinding?.acqTouchInterceptor?.isVisible = isLoading
    }

    private fun processChallengeResult(
        appBaseChallengeResult: AppBaseChallengeResult,
        card: CardResult
    ) {
        when (appBaseChallengeResult) {
            AppBaseChallengeResult.Loading -> {
                viewModel.handleEvent(
                    AttachCardViewModel.Event.ShowChallengeNotifiaction
                )
            }

            is AppBaseChallengeResult.TimeOut -> {
                viewModel.handleEvent(
                    AttachCardViewModel.Event.ChallengeTimeoutNotification(appBaseChallengeResult.error)
                )
            }

            is AppBaseChallengeResult.Success -> {
                viewModel.handleEvent(
                    AttachCardViewModel.Event.RequestGetCardState(requestKey = appBaseChallengeResult.requestKey)
                )
            }

            else -> sendResult(
                Result.ChallengeResult(
                    result = appBaseChallengeResult,
                    card = card
                )
            )
        }
    }

    private fun sendResult(result: Result) {
        parentFragmentManager.setFragmentResult(
            FRAGMENT_RESULT_KEY, bundleOf(
                FRAGMENT_RESULT_BUNDLE_KEY to result
            )
        )
    }

    companion object {
        const val FRAGMENT_RESULT_KEY = "AttachCardFragment.FRAGMENT_RESULT_KEY"
        const val FRAGMENT_RESULT_BUNDLE_KEY = "AttachCardFragment.FRAGMENT_RESULT_BUNDLE_KEY"

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (Result?) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(
                FRAGMENT_RESULT_KEY,
                lifecycleOwner
            ) { _, bundle ->
                listener.invoke(bundle.getParcelable(FRAGMENT_RESULT_BUNDLE_KEY))
            }
        }
    }

    sealed interface Result : Parcelable {
        @Parcelize
        data class CardAdded(val card: CardResult) : Result

        @Parcelize
        data object Cancelled : Result

        @Parcelize
        data class TimeOut(
            val error: Throwable,
            val panSuffix: String?,
        ) : Result

        @Parcelize
        data class Error(
            val error: Throwable,
            val errorCode: Int? = null,
            val panSuffix: String?,
        ) : Result

        @Parcelize
        data class ThreeDsRequired(
            val data: ThreeDsData,
            val panSuffix: String,
            val options: AttachCardOptions,
        ) : Result

        @Parcelize
        data class ChallengeResult(
            val result: AppBaseChallengeResult,
            val card: CardResult
        ) : Result
    }
}
