/*
 * Copyright © 2020 Tinkoff Bank
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package ru.tinkoff.acquiring.sdk.ui.fragments

import android.annotation.SuppressLint
import android.content.Context
import android.content.Intent
import android.content.res.Configuration
import android.graphics.Bitmap
import android.graphics.Point
import android.net.Uri
import android.os.Bundle
import android.os.Parcelable
import android.text.Spannable
import android.text.SpannableString
import android.text.method.ScrollingMovementMethod
import android.text.style.ForegroundColorSpan
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.WindowManager
import android.webkit.WebView
import android.webkit.WebViewClient
import android.widget.TextView
import androidx.core.content.ContextCompat
import androidx.core.os.bundleOf
import androidx.core.view.isVisible
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.navigation.fragment.navArgs
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import kotlinx.parcelize.Parcelize
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.databinding.AcqFragmentDynamicQrBinding
import ru.tinkoff.acquiring.sdk.localization.ASDKString
import ru.tinkoff.acquiring.sdk.ui.customview.NotificationDialog
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData
import ru.tinkoff.acquiring.sdk.viewmodel.DynamicQrCodeViewModel

/**
 * @author Mariya Chernyadieva
 */
internal open class DynamicQrCodeFragment : Fragment(), IsolatedKoinComponent {

    private var viewBinding: AcqFragmentDynamicQrBinding? = null
    private val args: DynamicQrCodeFragmentArgs by navArgs()


    protected val viewModel by viewModel<DynamicQrCodeViewModel> {
        parametersOf(args.options)
    }

    private var progressDialog: NotificationDialog? = null

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View? {
        val binding = AcqFragmentDynamicQrBinding.inflate(layoutInflater, container, false)
        val view = binding.root
        this.viewBinding = binding
        val webViewContainer = binding.acqQrWebViewContainer.root

        val webViewContainerPaddings = webViewContainer.paddingLeft + webViewContainer.paddingRight

        val screenScale =
            if (resources.configuration.orientation == Configuration.ORIENTATION_PORTRAIT) {
                getScreenScale().x
            } else {
                getScreenScale().y
            }

        viewBinding?.acqQrWebViewContainer?.acqStaticQrWv?.run {
            setInitialScale(screenScale / 2 - webViewContainerPaddings)
            isVerticalScrollBarEnabled = false
        }

        webViewContainer.layoutParams.height =
            screenScale - webViewContainerPaddings - webViewContainer.paddingTop

        binding.acqQrShare.setOnClickListener {
            viewModel.handleEvent(DynamicQrCodeViewModel.Event.ShareButtonClick)
        }

        return view
    }

    override fun onResume() {
        super.onResume()

        viewModel.handleEvent(DynamicQrCodeViewModel.Event.Resume)
    }

    private fun getScreenScale(): Point {
        val display =
            (requireActivity().getSystemService(Context.WINDOW_SERVICE) as WindowManager).defaultDisplay
        val point = Point()
        display.getSize(point)
        return point
    }

    private fun handleQrLinkResult(url: String) {
        val intent = Intent(Intent.ACTION_VIEW)
        intent.data = Uri.parse(url)
        startActivity(intent)
    }

    private fun handleLoadState(isLoading: Boolean) {
        if (isLoading) {
            viewBinding?.acqContent?.visibility = View.INVISIBLE
            viewBinding?.acqProgressbar?.root?.visibility = View.VISIBLE
        } else {
            viewBinding?.acqContent?.visibility = View.VISIBLE
            viewBinding?.acqProgressbar?.root?.visibility = View.INVISIBLE
        }
    }

    private fun handleQrResult(imageSvg: String) {
        val mimeType = "text/html"
        val encoding = "UTF-8"
        viewBinding?.acqQrWebViewContainer?.acqStaticQrWv?.loadDataWithBaseURL(
            "",
            "<html style=\"background: #F6F7F8;\"><center>$imageSvg</center></html>",
            mimeType,
            encoding,
            ""
        )

        viewBinding?.acqQrWebViewContainer?.acqStaticQrWv?.webViewClient =
            object : WebViewClient() {
                override fun onPageStarted(view: WebView?, url: String?, favicon: Bitmap?) {
                    super.onPageStarted(view, url, favicon)
                    viewBinding?.acqProgressbar?.root?.visibility = View.VISIBLE
                }

                override fun onPageFinished(view: WebView, url: String) {
                    super.onPageFinished(view, url)
                    viewBinding?.acqProgressbar?.root?.visibility = View.GONE
                    viewBinding?.acqContent?.visibility = View.VISIBLE
                }
            }
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        viewBinding?.acqQrTvAmountLabel?.text = getString(ASDKString.acq_pay_title)

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.stateFlow.collectLatest {
                    handleLoadState(it.isLoading)

                    if (it.qrImage != null) {
                        handleQrResult(it.qrImage)
                    }

                    if (it.error != null) {
                        viewBinding?.acqProgressbar?.root?.visibility = View.GONE
                        progressDialog?.dismiss()
                    }

                    if (it.showProgressDialog) {
                        if (progressDialog == null) {
                            progressDialog = NotificationDialog(requireContext()).apply {
                                showProgress()
                                show()
                            }
                        }
                        progressDialog?.show()
                    } else {
                        progressDialog?.dismiss()
                    }

                    it.paymentOptions?.order?.run {
                        viewBinding?.acqQrTvAmount?.text =
                            modifySpan(amount.toHumanReadableString())
                        viewBinding?.acqQrTvOrderTitle?.apply {
                            visibility = if (title.isNullOrBlank()) View.GONE else View.VISIBLE
                            text = title
                        }
                        viewBinding?.acqQrTvOrderDescription?.apply {
                            visibility =
                                if (description.isNullOrBlank()) View.GONE else View.VISIBLE
                            text = description
                            resolveScroll()
                        }
                    }

                    val notification = it.notification
                    if (notification != null) {
                        val data = notification.toStatusViewData(requireContext())
                        viewBinding?.acqStatusViewContainer?.acqStatusView?.fill(data)
                        viewBinding?.acqStatusViewContainer?.root?.isVisible = true
                    } else {
                        viewBinding?.acqStatusViewContainer?.root?.isVisible = false
                    }
                }
            }
        }

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.RESUMED) {
                viewModel.commandFlow.collect {
                    when (it) {
                        is DynamicQrCodeViewModel.Command.OpenLink -> {
                            handleQrLinkResult(it.url)
                        }

                        is DynamicQrCodeViewModel.Command.ReturnSuccessResult -> returnResult(
                            Result.Success(
                                it.paymentId
                            )
                        )

                        DynamicQrCodeViewModel.Command.ReturnCancelResult -> returnResult(Result.Cancel)
                        is DynamicQrCodeViewModel.Command.ReturnErrorResult -> returnResult(
                            Result.Failed(
                                error = it.error,
                                paymentId = it.paymentId
                            )
                        )
                    }
                }
            }
        }
    }

    override fun onDestroyView() {
        super.onDestroyView()
        progressDialog = null
    }

    @SuppressLint("ClickableViewAccessibility")
    private fun TextView.resolveScroll() {
        movementMethod = ScrollingMovementMethod()
        setOnTouchListener { _, _ ->
            val canScroll = canScrollVertically(1) || canScrollVertically(-1)
            parent.requestDisallowInterceptTouchEvent(canScroll)
            false
        }
    }

    private fun modifySpan(amount: String): CharSequence {
        val amountSpan = SpannableString(amount)
        val commaIndex = amount.indexOf(",")

        return if (commaIndex < 0) {
            amount
        } else {
            val coinsColor = ContextCompat.getColor(requireContext(), R.color.acq_colorCoins)
            amountSpan.setSpan(
                ForegroundColorSpan(coinsColor),
                commaIndex + 1,
                amount.length,
                Spannable.SPAN_EXCLUSIVE_EXCLUSIVE
            )
            amountSpan
        }
    }

    private fun returnResult(result: Result) {
        parentFragmentManager.setFragmentResult(
            REQUEST_KEY, bundleOf(
                RESULT_KEY to result
            )
        )
    }

    companion object {
        private const val TAG = "DynamicQrCodeFragment"
        private const val REQUEST_KEY = "$TAG.REQUEST_KEY"
        private const val RESULT_KEY = "$TAG.RESULT_KEY"

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (Result) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(REQUEST_KEY, lifecycleOwner) { _, bundle ->
                bundle.getParcelable<Result>(RESULT_KEY)?.let(listener)
            }
        }
    }

    sealed interface Result : Parcelable {
        @Parcelize
        data object Cancel : Result

        @Parcelize
        data class Success(
            val paymentId: Long
        ) : Result

        @Parcelize
        data class Failed(
            val error: Throwable,
            val paymentId: Long? = null
        ) : Result
    }
}
