/*
 * Copyright © 2020 Tinkoff Bank
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package ru.tinkoff.acquiring.sdk.ui.fragments

import android.content.Context
import android.content.Intent
import android.content.res.Configuration
import android.graphics.Bitmap
import android.graphics.Point
import android.net.Uri
import android.os.Bundle
import android.os.Parcelable
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.WindowManager
import android.webkit.WebView
import android.webkit.WebViewClient
import androidx.activity.addCallback
import androidx.core.os.bundleOf
import androidx.core.view.isVisible
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import kotlinx.parcelize.Parcelize
import org.koin.androidx.viewmodel.ext.android.viewModel
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.databinding.AcqFragmentStaticQrBinding
import ru.tinkoff.acquiring.sdk.localization.ASDKString
import ru.tinkoff.acquiring.sdk.ui.customview.NotificationDialog
import ru.tinkoff.acquiring.sdk.utils.toStatusViewData
import ru.tinkoff.acquiring.sdk.viewmodel.StaticQrCodeViewModel

internal class StaticQrCodeFragment : Fragment(), IsolatedKoinComponent {

    private var viewBinding: AcqFragmentStaticQrBinding? = null

    private val viewModel: StaticQrCodeViewModel by viewModel()

    private var progressDialog: NotificationDialog? = null

    override fun onAttach(context: Context) {
        super.onAttach(context)
        requireActivity().onBackPressedDispatcher.addCallback {
            viewModel.handleEvent(StaticQrCodeViewModel.Event.BackPressed)
        }
    }

    override fun onCreateView(
        inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?
    ): View {
        val binding = AcqFragmentStaticQrBinding.inflate(inflater, container, false)
        this.viewBinding = binding
        val view = binding.root
        val webViewContainer = binding.acqQrWebViewContainer.root
        val qrWebView = binding.acqQrWebViewContainer.acqStaticQrWv

        val webViewContainerPaddings = webViewContainer.paddingLeft + webViewContainer.paddingRight

        val screenScale =
            if (resources.configuration.orientation == Configuration.ORIENTATION_PORTRAIT) {
                getScreenScale().x
            } else {
                getScreenScale().y
            }

        qrWebView.run {
            setInitialScale(screenScale / 2 - webViewContainerPaddings)
            isVerticalScrollBarEnabled = false
        }

        webViewContainer.layoutParams.height =
            screenScale - webViewContainerPaddings - webViewContainer.paddingTop


        return view
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        viewBinding?.acqQrShare?.setOnClickListener {
            viewModel.handleEvent(StaticQrCodeViewModel.Event.ShareButtonClick)
        }

        viewBinding?.acqStaticQrTv?.text = getString(ASDKString.acq_qr_static_title)

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.stateFlow.collectLatest {
                    handleLoadState(it.isLoading)

                    if (it.qrImage != null) {
                        handleQrResult(it.qrImage)
                    }

                    if (it.error != null) {
                        viewBinding?.acqProgressbar?.root?.visibility = View.GONE
                        progressDialog?.dismiss()
                    }

                    if (it.showProgressDialog) {
                        if (progressDialog == null) {
                            progressDialog = NotificationDialog(requireContext())
                                .apply {
                                    showProgress()
                                    show()
                                }
                        } else {
                            progressDialog?.show()
                        }
                    } else {
                        progressDialog?.dismiss()
                    }

                    val notification = it.notification
                    if (notification != null) {
                        val data = notification.toStatusViewData(requireContext())
                        viewBinding?.acqStatusViewContainer?.acqStatusView?.fill(data)
                        viewBinding?.acqStatusViewContainer?.root?.isVisible = true
                    } else {
                        viewBinding?.acqStatusViewContainer?.root?.isVisible = false
                    }
                }
            }
        }

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.RESUMED) {
                viewModel.commandFlow.collect {
                    when (it) {
                        is StaticQrCodeViewModel.Command.OpenLink -> {
                            handleQrLinkResult(it.url)
                        }

                        StaticQrCodeViewModel.Command.ReturnCancelResult -> {
                            returnResult(Result.Cancel)
                        }

                        is StaticQrCodeViewModel.Command.ReturnErrorResult -> {
                            returnResult(Result.Failed(it.error))
                        }

                        is StaticQrCodeViewModel.Command.ReturnSuccessResult -> {
                            returnResult(Result.Success(paymentId = it.paymentId))
                        }
                    }
                }
            }
        }
    }

    override fun onResume() {
        super.onResume()

        viewModel.handleEvent(StaticQrCodeViewModel.Event.Resume)
    }

    override fun onDestroyView() {
        super.onDestroyView()
        progressDialog = null
    }

    private fun getScreenScale(): Point {
        val display =
            (requireActivity().getSystemService(Context.WINDOW_SERVICE) as WindowManager).defaultDisplay
        val point = Point()
        display.getSize(point)
        return point
    }

    private fun handleQrLinkResult(url: String) {
        progressDialog?.dismiss()

        val intent = Intent(Intent.ACTION_VIEW)
        intent.data = Uri.parse(url)
        startActivity(intent)
    }

    private fun handleLoadState(isLoading: Boolean) {
        if (isLoading) {
            viewBinding?.acqContent?.visibility = View.INVISIBLE
            viewBinding?.acqProgressbar?.root?.visibility = View.VISIBLE
        } else {
            viewBinding?.acqContent?.visibility = View.VISIBLE
            viewBinding?.acqProgressbar?.root?.visibility = View.INVISIBLE
        }
    }

    private fun handleQrResult(imageSvg: String) {
        val mimeType = "text/html"
        val encoding = "UTF-8"
        val webView = viewBinding?.acqQrWebViewContainer?.acqStaticQrWv
        webView?.loadDataWithBaseURL(
            "",
            "<html style=\"background: #F6F7F8;\"><center>$imageSvg</center></html>",
            mimeType,
            encoding,
            ""
        )

        webView?.webViewClient =
            object : WebViewClient() {
                override fun onPageStarted(view: WebView?, url: String?, favicon: Bitmap?) {
                    super.onPageStarted(view, url, favicon)
                    viewBinding?.acqProgressbar?.root?.visibility = View.VISIBLE
                }

                override fun onPageFinished(view: WebView, url: String) {
                    super.onPageFinished(view, url)
                    viewBinding?.acqProgressbar?.root?.visibility = View.GONE
                    viewBinding?.acqContent?.visibility = View.VISIBLE
                }
            }
    }

    private fun returnResult(result: Result) {
        parentFragmentManager.setFragmentResult(
            REQUEST_KEY, bundleOf(
                RESULT_KEY to result
            )
        )
    }

    companion object {
        private const val TAG = "StaticQrCodeFragment"
        private const val REQUEST_KEY = "$TAG.REQUEST_KEY"
        private const val RESULT_KEY = "$TAG.RESULT_KEY"

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (Result) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(REQUEST_KEY, lifecycleOwner) { _, bundle ->
                bundle.getParcelable<Result>(RESULT_KEY)?.let(listener)
            }
        }
    }

    sealed interface Result : Parcelable {
        @Parcelize
        data object Cancel : Result

        @Parcelize
        data class Success(
            val paymentId: Long
        ) : Result

        @Parcelize
        data class Failed(
            val error: Throwable
        ) : Result
    }
}
