/*
 * Copyright © 2020 Tinkoff Bank
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package ru.tinkoff.acquiring.sdk.ui.fragments.v2

import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.activity.addCallback
import androidx.core.os.bundleOf
import androidx.core.view.isVisible
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.SavedStateViewModelFactory
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import kotlinx.coroutines.launch
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.cardscanners.delegate.CardScannerWrapper
import ru.tinkoff.acquiring.sdk.cardscanners.delegate.ScannedCardResult
import ru.tinkoff.acquiring.sdk.databinding.AcqFragmentAttachCardV2Binding
import ru.tinkoff.acquiring.sdk.di.IsolatedKoinComponent
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringSdkException
import ru.tinkoff.acquiring.sdk.models.result.CardResult
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.v2.CardDataInputView
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeDelegate
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeDelegateImpl
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeResult
import ru.tinkoff.acquiring.sdk.ui.fragments.AttachCardFragment
import ru.tinkoff.acquiring.sdk.utils.AcqSnackBarHelper
import ru.tinkoff.acquiring.sdk.utils.KeyboardVisionUtils
import ru.tinkoff.acquiring.sdk.utils.setDisabledFieldAlpha

internal class AttachCardFragmentV2 : Fragment(),
    AppBaseChallengeDelegate by AppBaseChallengeDelegateImpl(), IsolatedKoinComponent {

    private val viewModel by viewModel<AttachCardViewModelV2> {
        parametersOf(
            SavedStateViewModelFactory(requireActivity().application, this, arguments)
        )
    }

    private var viewBinding: AcqFragmentAttachCardV2Binding? = null
    private var cardScannerWrapper: CardScannerWrapper? = null
    private var snackBarHelper: AcqSnackBarHelper? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        this.cardScannerWrapper = CardScannerWrapper(requireActivity(), ::onCardScannerResult)
    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        val binding = AcqFragmentAttachCardV2Binding.inflate(inflater, container, false)
        this.viewBinding = binding
        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        this.snackBarHelper = AcqSnackBarHelper(view)

        requireActivity().onBackPressedDispatcher.addCallback(viewLifecycleOwner) {
            viewModel.handleEvent(AttachCardViewModelV2.Event.BackButtonClicked)
        }

        initToolbar()

        viewBinding?.acqAttachBtnAttach?.setOnClickListener {
            hideKeyboard()
            viewModel.handleEvent(AttachCardViewModelV2.Event.AttachButtonClicked)
        }

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.stateFlow.collect {
                    updateCardDataInputFragment(it)

                    handleLoadState(it.isLoading)

                    viewBinding?.acqAttachBtnAttach?.isEnabled = it.payButtonEnabled
                }
            }
        }

        viewLifecycleOwner.lifecycleScope.launch {
            repeatOnLifecycle(Lifecycle.State.STARTED) {
                viewModel.commandFlow.collect {
                    when (it) {
                        is AttachCardViewModelV2.Command.FinishWithSuccess -> sendResult(
                            AttachCardFragment.Result.CardAdded(it.cardResult)
                        )

                        is AttachCardViewModelV2.Command.FinishWithError -> sendResult(
                            AttachCardFragment.Result.Error(
                                error = it.error,
                                errorCode = (it.error as? AcquiringSdkException)?.errorCode?.toIntOrNull(),
                                panSuffix = it.panSuffix,
                            )
                        )

                        is AttachCardViewModelV2.Command.Open3DSScreen -> sendResult(
                            AttachCardFragment.Result.ThreeDsRequired(
                                data = it.data,
                                panSuffix = it.panSuffix,
                                options = it.options
                            )
                        )

                        is AttachCardViewModelV2.Command.FinishWithTimeOut -> {
                            sendResult(AttachCardFragment.Result.TimeOut(
                                error = it.error,
                                panSuffix = it.panSuffix
                            ))
                        }

                        is AttachCardViewModelV2.Command.Start3DsChallenge -> {
                            initAppBaseChallengeDelegate(
                                requireActivity(),
                                AcquiringSdk(
                                    it.options.terminalKey,
                                    it.options.publicKey,
                                ),
                                viewLifecycleOwner.lifecycleScope
                            )
                            launchChallenge(
                                it.threeDsData,
                                it.transaction,
                                onResult = { result: AppBaseChallengeResult ->
                                    processChallengeResult(result, it.cardResult)
                                },
                            )
                        }

                        is AttachCardViewModelV2.Command.Cancel -> {
                            sendResult(AttachCardFragment.Result.Cancelled)
                        }

                        AttachCardViewModelV2.Command.RequestCardNumberFieldFocus -> {
                            setFocusToCardNumber()
                        }

                        is AttachCardViewModelV2.Command.FillCardData -> {
                            viewBinding?.cardDataInput?.apply {
                                setNumber(it.cardNumber)
                                setExpireDate(it.expireDate)
                                setCvc(it.cvc)
                            }
                        }
                        AttachCardViewModelV2.Command.RequestCvcFieldFocus -> {
                            viewBinding?.cardDataInput?.focusCvcField()
                        }
                        AttachCardViewModelV2.Command.RequestExpireDateFieldFocus -> {
                            viewBinding?.cardDataInput?.focusExpireDateField()
                        }
                        is AttachCardViewModelV2.Command.ShowCardAddError -> {
                            showErrorToast(it.args, it.message)
                        }
                        AttachCardViewModelV2.Command.ClearFocus -> {
                            KeyboardVisionUtils.hideKeyboard(requireView())
                            viewBinding?.cardDataInput?.clearFocus()
                        }
                    }
                }
            }
        }
    }

    private fun showErrorToast(args: List<Any>, message: Int) {
        snackBarHelper?.showWithIcon(
            R.drawable.acq_ic_alert_ngative,
            if (args.isNotEmpty()) {
                getString(message, *args.toTypedArray())
            } else {
                getString(message)
            }
        )
    }

    override fun onResume() {
        super.onResume()
        viewModel.handleEvent(AttachCardViewModelV2.Event.Resume)
    }

    private fun setFocusToCardNumber() {
        viewBinding?.cardDataInput?.focusNumberField()
    }

    private fun onCardScannerResult(result: ScannedCardResult) {
        when(result) {
            ScannedCardResult.Cancel,
            is ScannedCardResult.Failure -> Unit
            is ScannedCardResult.Success -> {
                viewModel.handleEvent(AttachCardViewModelV2.Event.CardScanned(
                    cardNumber = result.data.cardNumber,
                    expireDate = result.data.expireDate
                ))
            }
        }
    }

    private fun updateCardDataInputFragment(state: AttachCardViewModelV2.State) {
        val binding = viewBinding ?: return

        val options = state.options

        val cardDataInput = binding.cardDataInput
        cardDataInput.listener = null
        cardScannerWrapper?.cameraCardScannerContract = options.features.cameraCardScannerContract
        cardDataInput.isScanEnabled = cardScannerWrapper?.isEnabled == true
        cardDataInput.useSecuredKeyboard = options.features.useSecureKeyboard
        cardDataInput.isVisible = true
        cardDataInput.showCvcError(state.isValidCvc == false)
        cardDataInput.showCardNumberError(state.isValidCardNumber == false)
        cardDataInput.expireDateError(state.isValidExpireDate == false)
        cardDataInput.listener = object: CardDataInputView.Listener {
            override fun onCardNumberChanged(cardNumber: String) {
                viewModel.handleEvent(AttachCardViewModelV2.Event.CardNumberChanged(cardNumber))
            }

            override fun onExpireDateChanged(expireDate: String) {
                viewModel.handleEvent(AttachCardViewModelV2.Event.ExpireDateChanged(expireDate))
            }

            override fun onCvcChanged(cvc: String) {
                viewModel.handleEvent(AttachCardViewModelV2.Event.CvcChanged(cvc))
            }

            override fun onScanButtonClick() {
                cardScannerWrapper?.start()
            }

            override fun onFocusChanged(
                isCardNumberFocused: Boolean,
                isExpireDateFocused: Boolean,
                isCvcFocused: Boolean
            ) {
                viewModel.handleEvent(
                    AttachCardViewModelV2.Event.CardFocusChanged(
                    isCardNumberFocused = isCardNumberFocused,
                    isExpireDateFocused = isExpireDateFocused,
                    isCvcFocused= isCvcFocused,
                ))
            }
        }
    }

    private fun hideKeyboard() {
        requireActivity().currentFocus?.clearFocus()
        KeyboardVisionUtils.hideKeyboard(requireView())
    }

    override fun onDestroyView() {
        super.onDestroyView()
        viewBinding = null
        snackBarHelper = null
    }

    private fun initToolbar() {
        viewBinding?.acqToolbar?.apply {
            setTitle(R.string.acq_cardlist_addcard_title)
            setNavigationOnClickListener {
                viewModel.handleEvent(AttachCardViewModelV2.Event.BackButtonClicked)
            }
        }
    }

    private fun handleLoadState(isLoading: Boolean) {
        val binding = viewBinding ?: return
        binding.acqAttachBtnAttach.isLoading = isLoading
        binding.acqTouchInterceptor.isVisible = isLoading
        binding.cardDataInput.apply {
            setDisabledFieldAlpha(isLoading)
            isEnabled = !isLoading
        }
    }

    private fun processChallengeResult(
        appBaseChallengeResult: AppBaseChallengeResult,
        card: CardResult
    ) {
        when (appBaseChallengeResult) {
            AppBaseChallengeResult.Loading -> {
                viewModel.handleEvent(
                    AttachCardViewModelV2.Event.ShowChallengeNotifiaction
                )
            }

            is AppBaseChallengeResult.TimeOut -> {
                viewModel.handleEvent(
                    AttachCardViewModelV2.Event.ChallengeTimeoutNotification(appBaseChallengeResult.error)
                )
            }

            is AppBaseChallengeResult.Success -> {
                viewModel.handleEvent(
                    AttachCardViewModelV2.Event.RequestGetCardState(requestKey = appBaseChallengeResult.requestKey)
                )
            }

            else -> sendResult(
                AttachCardFragment.Result.ChallengeResult(
                    result = appBaseChallengeResult,
                    card = card
                )
            )
        }
    }

    private fun sendResult(result: AttachCardFragment.Result) {
        parentFragmentManager.setFragmentResult(
            FRAGMENT_RESULT_KEY, bundleOf(
                FRAGMENT_RESULT_BUNDLE_KEY to result
            )
        )
    }

    companion object {
        const val FRAGMENT_RESULT_KEY = "AttachCardFragmentV2.FRAGMENT_RESULT_KEY"
        const val FRAGMENT_RESULT_BUNDLE_KEY = "AttachCardFragmentV2.FRAGMENT_RESULT_BUNDLE_KEY"

        fun registerResultListener(
            fragmentManager: FragmentManager,
            lifecycleOwner: LifecycleOwner,
            listener: (AttachCardFragment.Result?) -> Unit
        ) {
            fragmentManager.setFragmentResultListener(
                FRAGMENT_RESULT_KEY,
                lifecycleOwner
            ) { _, bundle ->
                listener.invoke(bundle.getParcelable(FRAGMENT_RESULT_BUNDLE_KEY))
            }
        }
    }
}
