package ru.tinkoff.acquiring.sdk.ui.fragments.v2

import android.app.Application
import android.os.Build
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.BuildConfig
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringApiException
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringSdkException
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.enums.CheckType
import ru.tinkoff.acquiring.sdk.models.enums.ResponseStatus
import ru.tinkoff.acquiring.sdk.models.options.screen.AttachCardOptions
import ru.tinkoff.acquiring.sdk.models.paysources.CardData
import ru.tinkoff.acquiring.sdk.models.result.CardResult
import ru.tinkoff.acquiring.sdk.network.AcquiringApi.API_ERROR_CODE_CARD_ALREADY_ATTACHED
import ru.tinkoff.acquiring.sdk.redesign.common.carddatainput.CardNumberFormatter
import ru.tinkoff.acquiring.sdk.responses.AttachCardResponse
import ru.tinkoff.acquiring.sdk.responses.Check3dsVersionResponse
import ru.tinkoff.acquiring.sdk.threeds.AppBaseProcessResult
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsAppBaseProcess
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsAppBasedTransaction
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsHelper
import ru.tinkoff.acquiring.sdk.toggles.FeatureToggleManager
import ru.tinkoff.acquiring.sdk.toggles.toggles.AppBaseFeatureToggle
import ru.tinkoff.acquiring.sdk.toggles.withToggleInfo
import ru.tinkoff.acquiring.sdk.ui.delegate.CardDataInputDelegate
import ru.tinkoff.acquiring.sdk.utils.panSuffix

/**
 * @author Mariya Chernyadieva
 */
internal class AttachCardViewModelV2(
    application: Application,
    savedStateHandle: SavedStateHandle,
    private val dsAppBaseProcess: ThreeDsAppBaseProcess,
    private val toggleManager: FeatureToggleManager,
    private val cardDataInputDelegate: CardDataInputDelegate
) : AndroidViewModel(application) {

    private var firstTimeResume: Boolean = true
    private var args: AttachCardFragmentV2Args =
        AttachCardFragmentV2Args.fromSavedStateHandle(savedStateHandle)
    private val options: AttachCardOptions = args.attachCardOptions

    private val sdk = AcquiringSdk(
        publicKey = options.publicKey,
        terminalKey = options.terminalKey,
    )
    private val _stateFlow = MutableStateFlow(
        State(
            options = options
        )
    )
    val stateFlow = _stateFlow.asStateFlow()
    private val _commandFlow = MutableSharedFlow<Command>()
    val commandFlow = _commandFlow.asSharedFlow()

    fun handleEvent(event: Event) {
        when (event) {
            Event.AttachButtonClicked -> onAttachButtonClicked()
            Event.BackButtonClicked -> onBackButtonClicked()
            Event.ShowChallengeNotifiaction -> Unit

            is Event.ChallengeTimeoutNotification -> {
                showErrorToast(event.error)
                _stateFlow.update {
                    it.copy(
                        isLoading = false
                    )
                }
            }

            is Event.RequestGetCardState -> requestAddCardState(requestKey = event.requestKey)

            Event.Resume -> onViewResume()
            is Event.CardFocusChanged -> onCardFocusChanged(
                cardNumberFocused = event.isCardNumberFocused,
                expireDateFocused = event.isExpireDateFocused,
                cvcFocused = event.isCvcFocused
            )
            is Event.CvcChanged -> onCvcChanged(event.cvc)
            is Event.ExpireDateChanged -> onCardExpireDateChanged(event.expireDate)
            is Event.CardNumberChanged -> onCardNumberChanged(event.cardNumber)
            is Event.CardScanned -> onCardScanned(event.cardNumber, event.expireDate)
        }
    }

    private fun showErrorToast(error: Throwable) {
        var message = R.string.acq_cardlist_snackbar_add_error_nopan
        var args = emptyList<Any>()
        val panSuffix = stateFlow.value.cardNumber?.panSuffix()
        when {
            (error as? AcquiringApiException)?.errorCode == API_ERROR_CODE_CARD_ALREADY_ATTACHED -> {
                message = R.string.acq_cardlist_snackbar_add_error_already_attached
            }

            panSuffix != null -> {
                message = R.string.acq_cardlist_snackbar_add_error
                args = listOf(panSuffix.panSuffix())
            }
        }
        viewModelScope.launch {
            _commandFlow.emit(
                Command.ShowCardAddError(
                    message = message,
                    args = args
                )
            )
        }
    }

    private fun onCvcChanged(cvc: String) {
        validateCvc(
            cvc = cvc,
            isCvcFocused = stateFlow.value.isCvcFocused
        )
    }

    private fun onCardExpireDateChanged(expireDate:String) {
        validateExpireDateAndFocusNextField(expireDate, true)
    }

    private fun onCardNumberChanged(cardNumber: String) {
        validateCardNumberAndGoToNextField(
            cardNumber = cardNumber,
            gotoNextField = true,
            forceHideError = false,
            isFocused = stateFlow.value.isCardNumberFocused,
        )
    }

    private fun onCardScanned(cardNumber: String?, expireDate: String?) {
        cardDataInputDelegate.onCardScanned(
            cardNumber = cardNumber,
            expireDate = expireDate,
            updateState = { cardNumber, expireDate ->
                viewModelScope.launch {
                    _commandFlow.emit(
                        Command.FillCardData(
                            cardNumber = cardNumber,
                            expireDate = expireDate
                        )
                    )
                }
            },
            requestCvcFocus = {
                viewModelScope.launch {
                    _commandFlow.emit(Command.RequestCvcFieldFocus)
                }
            },
            requestExpireDateFocus = {
                viewModelScope.launch {
                    _commandFlow.emit(Command.RequestExpireDateFieldFocus)
                }
            }
        )
    }

    private fun onCardFocusChanged(
        cardNumberFocused: Boolean,
        expireDateFocused: Boolean,
        cvcFocused: Boolean
    ) {
        val currentState = stateFlow.value
        _stateFlow.update { it.copy(
            isCardNumberFocused = cardNumberFocused,
            isExpireDateFocused = expireDateFocused,
            isCvcFocused = cvcFocused
        ) }
        if (currentState.isCardNumberFocused != cardNumberFocused) {
            validateCardNumberAndGoToNextField(
                cardNumber = currentState.cardNumber.orEmpty(),
                gotoNextField = false,
                forceHideError = cardNumberFocused,
                isFocused = stateFlow.value.isCardNumberFocused,
            )
        }
        if (currentState.isExpireDateFocused != expireDateFocused) {
            validateExpireDateAndFocusNextField(
                expireDate = currentState.expireDate.orEmpty(),
                allowGoToNextField = false
            )
        }
        if (currentState.isCvcFocused != cvcFocused) {
            validateCvc(
                cvc =currentState.cvc.orEmpty(),
                isCvcFocused = cvcFocused
            )
        }
    }

    private fun validateCvc(cvc: String, isCvcFocused: Boolean) {
        cardDataInputDelegate.validateCvc(
            cvc = cvc,
            isFocused = isCvcFocused,
            updateState = { cvc, isCvcValid ->
                _stateFlow.update {
                    it.copy(
                        cvc = cvc,
                        isValidCvc = isCvcValid
                    )
                }
            }
        )
    }

    private fun validateExpireDateAndFocusNextField(expireDate: String, allowGoToNextField: Boolean ) {
        val isFocused = stateFlow.value.isExpireDateFocused
        cardDataInputDelegate.validateExpireDateAndFocusNextField(
            expireDate = expireDate,
            isFocused = isFocused,
            allowGoToNextField = allowGoToNextField,
            goToNextField = ::gotoCvcFieldOrClearFocus,
            updateState = { expireDate, isValidExpireDate ->
                _stateFlow.update {
                    it.copy(
                        expireDate = expireDate,
                        isValidExpireDate = isValidExpireDate
                    )
                }
            }
        )
    }

    private fun gotoCvcFieldOrClearFocus() {
        val cvc = _stateFlow.value.cvc
        if (cvc.isNullOrBlank()) {
            viewModelScope.launch {
                _commandFlow.emit(Command.RequestCvcFieldFocus)
            }
        } else {
            viewModelScope.launch {
                _commandFlow.emit(Command.ClearFocus)
            }
        }
    }

    private fun validateCardNumberAndGoToNextField(
        cardNumber: String,
        gotoNextField: Boolean,
        forceHideError: Boolean,
        isFocused: Boolean,
    ) {
        cardDataInputDelegate.validateCardNumberAndGoToNextField(
            cardNumber = cardNumber,
            allowGotoNextField = gotoNextField,
            forceHideError = forceHideError,
            isFocused = isFocused,
            updateState = { normalizedCardNumber, isValidCardNumber ->
                _stateFlow.update {
                    it.copy(
                        cardNumber = normalizedCardNumber,
                        isValidCardNumber = isValidCardNumber
                    )
                }
            },
            goToNextField = {
                viewModelScope.launch {
                    _commandFlow.emit(Command.RequestExpireDateFieldFocus)
                }
            }
        )
    }

    private fun onViewResume() {
        if (firstTimeResume) {
            onViewResumeFirstTime()
        } else {
            viewModelScope.launch {
                val currentState = _stateFlow.value
                _commandFlow.emit(Command.FillCardData(
                    cardNumber = currentState.cardNumber,
                    expireDate = currentState.expireDate,
                    cvc = currentState.cvc,
                ))
            }
        }
    }

    private fun onViewResumeFirstTime() {
        firstTimeResume = false
        viewModelScope.launch {
            _commandFlow.emit(Command.RequestCardNumberFieldFocus)
        }
    }

    private fun requestAddCardState(requestKey: String?) {
        viewModelScope.launch(Dispatchers.IO) {
            sdk.getAddCardState {
                this.requestKey = requestKey
            }.execute(onSuccess = { response ->
                if (response.status == ResponseStatus.COMPLETED) {
                    viewModelScope.launch {
                        _commandFlow.emit(
                            Command.FinishWithSuccess(
                                CardResult(
                                    cardId = response.cardId,
                                    panSuffix = stateFlow.value.cardNumber?.panSuffix()
                                )
                            )
                        )
                    }
                } else {
                    val throwable = AcquiringSdkException(
                        IllegalStateException("AsdkState = ${response.status}"),
                        errorCode = response.errorCode,
                    )
                    viewModelScope.launch {
                        _commandFlow.emit(
                            Command.FinishWithError(
                                error = throwable,
                                panSuffix = stateFlow.value.cardNumber?.panSuffix()
                            )
                        )
                    }
                }
            }, onFailure = {
                viewModelScope.launch {
                    _commandFlow.emit(
                        Command.FinishWithError(
                            error = it,
                            panSuffix = stateFlow.value.cardNumber?.panSuffix()
                        )
                    )
                }
            })
        }
    }


    private fun onAttachButtonClicked() {
        val attachCardOptions = options
        val customerOptions = attachCardOptions.customer
        val customerKey = requireNotNull(customerOptions.customerKey) { "Не указан customerKey" }
        val checkType = attachCardOptions.customer.checkType
        val data = customerOptions.data
        val cardData = getCardData()
        if (cardData != null) {
            startAttachCard(cardData, customerKey, checkType, data)
        }
    }

    private fun onBackButtonClicked() {
        sendCancel()
    }

    private fun attachCard(
        requestKey: String,
        data: Map<String, String>?,
        check3dsVersionResponse: Check3dsVersionResponse? = null,
        transaction: ThreeDsAppBasedTransaction? = null,
        paymentId: Long? = null
    ) {
        viewModelScope.launch(Dispatchers.IO) {
            val dataWithToggleInfo = data.withToggleInfo()
            val attachCardRequest = sdk.attachCard {
                this.requestKey = requestKey
                this.data = dataWithToggleInfo
                this.cardData = getCardData()
                if (check3dsVersionResponse?.is3DsVersionV2() == true) {
                    this.addContentHeader()
                    this.addUserAgentHeader()
                }
                this.sdkVersion = BuildConfig.ASDK_VERSION_NAME
                this.softwareVersion = Build.VERSION.SDK_INT.toString()
                this.deviceModel = Build.MODEL
            }
            try {
                val result = attachCardRequest.execute()
                handleAttachSuccess(result, check3dsVersionResponse, transaction, paymentId)
            } catch (e: Exception) {
                handleAttachError(e)
            }
        }
    }

    private fun getCardData(): CardData? {
        val currentState = stateFlow.value
        val cardNumber = currentState.cardNumber.takeIf { currentState.isValidCardNumber == true }
        val expireDate = currentState.expireDate.takeIf {currentState.isValidExpireDate == true }
        val cvc = currentState.cvc.takeIf { currentState.isValidCvc == true }
        return if (cardNumber != null && expireDate != null && cvc != null) {
            CardData(
                pan = CardNumberFormatter.normalize(cardNumber),
                expiryDate = expireDate,
                securityCode = cvc,
            )
        } else null
    }

    private fun handleAttachSuccess(
        it: AttachCardResponse,
        check3dsVersionResponse: Check3dsVersionResponse?,
        transaction: ThreeDsAppBasedTransaction? = null,
        paymentId: Long? = null,
    ) {
        _stateFlow.update {
            it.copy(
                isLoading = false
            )
        }
        when (it.status) {
            ResponseStatus.THREE_DS_CHECKING -> {
                viewModelScope.launch {
                    if (check3dsVersionResponse?.isAppBase() == true
                        && toggleManager.isEnabled(AppBaseFeatureToggle)
                    ) {
                        _commandFlow.emit(
                            Command.Start3DsChallenge(
                                threeDsData = ThreeDsData(
                                    acsUrl = it.acsUrl,
                                    paymentId = paymentId,
                                    requestKey = it.requestKey
                                ).apply {
                                    acsRefNumber = it.acsReferenceNumber
                                    acsTransId = it.acsTransId
                                    acsSignedContent = it.acsSignedContent
                                    tdsServerTransId =
                                        checkNotNull(check3dsVersionResponse.serverTransId)
                                },
                                transaction = requireNotNull(transaction),
                                options = options,
                                cardResult = CardResult(
                                    it.cardId,
                                    getCardData()?.pan?.panSuffix()
                                )
                            )
                        )
                    } else {
                        val threeDsData =
                            it.getThreeDsData(paymentSys = check3dsVersionResponse?.paymentSystem)
                        if (check3dsVersionResponse?.is3DsVersionV2() == true) {
                            ThreeDsHelper.CollectData.addExtraThreeDsData(
                                data = threeDsData,
                                acsTransId = checkNotNull(it.acsTransId),
                                serverTransId = checkNotNull(check3dsVersionResponse.serverTransId),
                                version = checkNotNull(check3dsVersionResponse.version)
                            )
                        }
                        _commandFlow.emit(
                            Command.Open3DSScreen(
                                data = threeDsData,
                                panSuffix = getCardData()?.pan?.panSuffix()!!,
                                options = options,
                            )
                        )
                    }
                }
            }

            null -> {
                viewModelScope.launch {
                    _commandFlow.emit(
                        Command.FinishWithSuccess(
                            cardResult = CardResult(it.cardId, getCardData()?.pan?.panSuffix())
                        )
                    )
                }
            }

            else -> {
                val e = AcquiringSdkException(
                    IllegalStateException("ResponseStatus = ${it.status}")
                )
                handleAttachError(e)
            }
        }
    }

    private fun handleAttachError(e: Exception) {
        AcquiringSdk.log(e)
        showErrorToast(e)
        _stateFlow.update {
            it.copy(
                isLoading = false,
            )
        }
    }

    private fun sendCancel() {
        viewModelScope.launch {
            _commandFlow.emit(Command.Cancel)
        }
    }

    private fun startAttachCard(
        cardData: CardData,
        customerKey: String,
        checkType: String,
        data: Map<String, String>?
    ) {
        _stateFlow.update {
            it.copy(isLoading = true)
        }

        viewModelScope.launch(Dispatchers.IO) {
            val addCardRequest = sdk.addCard {
                this.customerKey = customerKey
                this.checkType = checkType
            }

            try {
                val it = addCardRequest.execute()
                check3dsVersionIfNeed(cardData, it.paymentId, it.requestKey!!, data)
            } catch (e: Exception) {
                AcquiringSdk.log(e)
                handleAttachError(e)
            }
        }
    }

    private fun check3dsVersionIfNeed(
        cardData: CardData,
        paymentId: Long?,
        requestKey: String,
        data: Map<String, String>?
    ) {
        when (options.customer.checkType) {
            CheckType.THREE_DS.toString(),
            CheckType.THREE_DS_HOLD.toString(),
            CheckType.AUTO -> {
                viewModelScope.launch(Dispatchers.IO) {
                    try {
                        val check3DsRequest = sdk.check3DsVersion {
                            this.paymentId = paymentId
                            this.paymentSource = cardData
                        }
                        val result = check3DsRequest.execute()

                        add3dsVersionToContext(result)

                        when {
                            result.isAppBase() && toggleManager.isEnabled(AppBaseFeatureToggle) -> {
                                when (val appBaseResult = dsAppBaseProcess.collectInfo(result)) {

                                    is AppBaseProcessResult.Error -> {
                                        handleAttachError(appBaseResult.throwable)
                                    }

                                    is AppBaseProcessResult.Success -> {
                                        attachCard(
                                            requestKey,
                                            appBaseResult.threedsData + (data ?: mapOf()),
                                            result,
                                            appBaseResult.transaction,
                                            paymentId
                                        )
                                    }
                                }
                            }

                            result.is3DsVersionV2() -> {
                                val check3dsMap: MutableMap<String, String> = get3DsParams(result)
                                attachCard(
                                    requestKey,
                                    check3dsMap + (data ?: mapOf()),
                                    result,
                                )
                            }

                            else -> {
                                attachCard(requestKey, data, result)
                            }
                        }
                    } catch (e: Exception) {
                        handleAttachError(e)
                    }
                }
            }

            else -> {
                attachCard(requestKey, data)
            }
        }
    }

    private fun add3dsVersionToContext(result: Check3dsVersionResponse) {
        try {
            val threeDsVersion = when {
                result.isAppBase() -> FeatureToggleManager.FeatureToggleContext.VERSION_2_APP
                result.is3DsVersionV2() -> FeatureToggleManager.FeatureToggleContext.VERSION_2_BROWSER
                else -> FeatureToggleManager.FeatureToggleContext.VERSION_1
            }
            viewModelScope.launch {
                toggleManager.updateFeatureToggleContext {
                    it.copy(
                        threeDsVersion = threeDsVersion,
                        appVersion = AcquiringSdk.appVersion,
                    )
                }
            }
        } catch (e: Exception) {
            AcquiringSdk.log(e)
        }
    }

    private suspend fun get3DsParams(result: Check3dsVersionResponse): MutableMap<String, String> {
        var check3dsMap: MutableMap<String, String>
        withContext(Dispatchers.Main) {
            check3dsMap = ThreeDsHelper.CollectData.invoke(getApplication(), result)
            ThreeDsHelper.CollectData.addExtraData(check3dsMap, result)
        }
        return check3dsMap
    }

    sealed interface Event {
        data object Resume : Event
        data object AttachButtonClicked : Event
        data object BackButtonClicked : Event
        data object ShowChallengeNotifiaction : Event
        class ChallengeTimeoutNotification(
            val error: Throwable
        ) : Event

        class RequestGetCardState(
            val requestKey: String? = null
        ) : Event

        class CardNumberChanged(
            val cardNumber: String
        ) : Event

        class ExpireDateChanged(
            val expireDate: String
        ) : Event

        class CvcChanged(
            val cvc: String
        ) : Event

        class CardScanned(
            val cardNumber: String,
            val expireDate: String,
        ) : Event

        class CardFocusChanged(
            val isCardNumberFocused: Boolean,
            val isExpireDateFocused: Boolean,
            val isCvcFocused: Boolean
        ) : Event
    }

    data class State(
        val options: AttachCardOptions,
        val isLoading: Boolean = false,
        val cardNumber: String? = null,
        val isCardNumberFocused: Boolean = false,
        val isValidCardNumber: Boolean? = null,
        val expireDate: String? = null,
        val isExpireDateFocused: Boolean = false,
        val isValidExpireDate: Boolean? = null,
        val cvc: String? = null,
        val isCvcFocused: Boolean = false,
        val isValidCvc: Boolean? = null,
    ) {
        val payButtonEnabled: Boolean
            get() = isValidCardNumber ==true
                    && isValidExpireDate == true
                    && isValidCvc == true
    }

    sealed interface Command {
        class FillCardData(
            val cardNumber: String?,
            val expireDate: String?,
            val cvc: String? = null,
        ) : Command

        data class FinishWithSuccess(
            val cardResult: CardResult
        ) : Command

        data class Open3DSScreen(
            val data: ThreeDsData,
            val panSuffix: String,
            val options: AttachCardOptions,
        ) : Command

        data class FinishWithTimeOut(
            val error: Throwable,
            val panSuffix: String?,
        ) : Command

        data class Start3DsChallenge(
            val transaction: ThreeDsAppBasedTransaction,
            val threeDsData: ThreeDsData,
            val options: AttachCardOptions,
            val cardResult: CardResult
        ) : Command

        data class FinishWithError(
            val error: Exception,
            val panSuffix: String?
        ) : Command

        data class ShowCardAddError(
            val message: Int,
            val args: List<Any> = emptyList(),
        ) : Command

        data object Cancel : Command
        data object RequestCardNumberFieldFocus : Command
        data object RequestExpireDateFieldFocus : Command
        data object RequestCvcFieldFocus : Command
        data object ClearFocus : Command
    }
}
