/*
 * Copyright © 2020 Tinkoff Bank
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package ru.tinkoff.acquiring.sdk.utils

import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers.Default
import kotlinx.coroutines.Dispatchers.IO
import kotlinx.coroutines.Dispatchers.Main
import kotlinx.coroutines.Job
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * @author Mariya Chernyadieva
 */
// TODO нужны только диспы отсюда
internal class CoroutineManager(
    private val exceptionHandler: (Throwable) -> Unit = {},
    val io: CoroutineDispatcher = IO,
    val main: CoroutineDispatcher = Main,
    val default: CoroutineDispatcher = Default,
) {

    constructor(dispatcher: CoroutineDispatcher) : this(
        io = dispatcher,
        default = dispatcher,
        main = dispatcher
    )

    private val job = SupervisorJob()
    private val coroutineExceptionHandler =
        CoroutineExceptionHandler { _, throwable -> launchOnMain { exceptionHandler(throwable) } }
    private val coroutineScope = CoroutineScope(Main + coroutineExceptionHandler + job)
    private val disposableSet = hashSetOf<Disposable>()

    fun cancelAll() {
        disposableSet.forEach {
            it.dispose()
        }
        job.cancel()
    }

    suspend fun withMain(block: suspend CoroutineScope.() -> Unit) {
        withContext(main) {
            block.invoke(this)
        }
    }

    fun launchOnBackground(block: suspend CoroutineScope.() -> Unit): Job {
        return coroutineScope.launch(io) {
            block.invoke(this)
        }
    }

    private fun launchOnMain(block: suspend CoroutineScope.() -> Unit): Job {
        return coroutineScope.launch(main) {
            block.invoke(this)
        }
    }
}
