package ru.tinkoff.acquiring.sdk.utils

import ru.tinkoff.acquiring.sdk.AcquiringSdkContext
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.RunMode
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringNetworkConnectException
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringNetworkSSLException
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringNetworkTimeoutException
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringSdkTimeoutException
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.ui.customview.status.StatusViewData
import java.net.ConnectException

/**
 * @author s.y.biryukov
 */
object NotificationFactory {
    fun getContentLoadingErrorNotification(
        exception: Throwable,
        onClickFactory: (e: Throwable) -> (() -> Unit)?
    ): Notification {
        return when (exception) {
            is AcquiringNetworkTimeoutException -> Notification(
                type = StatusViewData.Type.ERROR,
                description = R.string.acq_generic_stubnet_title,
                button = R.string.acq_generic_button_repeat,
                onClick = onClickFactory(exception)
            )

            is AcquiringNetworkConnectException,
            is AcquiringNetworkSSLException -> getOfflineNotification(exception, onClickFactory)

            else -> getErrorCommonNotification(exception, onClickFactory)
        }
    }

    fun getOfflineNotification(
        exception: Throwable = ConnectException(),
        onClickFactory: (e: Throwable) -> (() -> Unit)?
    ) = Notification(
        type = StatusViewData.Type.OFFLINE,
        title = R.string.acq_generic_stubnet_title,
        description = R.string.acq_generic_stubnet_description,
        button = R.string.acq_generic_button_stubnet,
        onClick = onClickFactory(exception)
    )

    fun getErrorCommonNotification(
        exception: Throwable,
        onClickFactory: (e: Throwable) -> (() -> Unit)?
    ) = Notification(
        type = StatusViewData.Type.ALARM,
        title = R.string.acq_generic_alert_label,
        description = R.string.acq_generic_stub_description,
        button = R.string.acq_generic_alert_access,
        onClick = onClickFactory(exception)
    )

    fun getPaymentErrorNotification(
        exception: Throwable,
        onClickFactory: (e: Throwable) -> (() -> Unit)?,
        options: Notification.Options
    ): Notification {
        return when (exception) {
            is AcquiringSdkTimeoutException -> getPaymentTimeoutErrorNotification(
                exception,
                onClickFactory
            )
            else -> getPaymentErrorCommonNotification(
                exception,
                onClickFactory,
                options,
            )
        }
    }

    internal fun getNotificationOptions(sdkContext: AcquiringSdkContext): Notification.Options {
        return Notification.Options(
            isTerminal = sdkContext.runMode != RunMode.FORM
        )
    }

    private fun getPaymentErrorCommonNotification(
        exception: Throwable,
        onClickFactory: (e: Throwable) -> (() -> Unit)?,
        options: Notification.Options
    ): Notification {
        return if (options.isTerminal) {
            Notification(
                type = StatusViewData.Type.ERROR,
                title = R.string.acq_commonsheet_payment_failed_title,
                description = R.string.acq_commonsheet_payment_failed_description_final,
                button = R.string.acq_commonsheet_payment_failed_primary_button_final,
                onClick = onClickFactory(exception)
            )
        } else {
            Notification(
                type = StatusViewData.Type.ERROR,
                title = R.string.acq_commonsheet_payment_failed_title,
                description = R.string.acq_commonsheet_payment_failed_description_retry,
                button = R.string.acq_commonsheet_payment_failed_primary_button_retry,
                onClick = onClickFactory(exception)
            )
        }
    }

    fun getPaymentTimeoutErrorNotification(
        exception: Throwable,
        onClickFactory: (e: Throwable) -> (() -> Unit)?
    ): Notification {
        return Notification(
            type = StatusViewData.Type.PAYMENT_TIMEOUT,
            title = null,
            description = R.string.acq_commonsheet_timeout_failed_title,
            button = R.string.acq_commonsheet_payment_failed_primary_button_retry,
            onClick = onClickFactory(exception),
        )
    }

    fun getPaymentSuccessCommonNotification(
        amount: Money?,
        onClick: (() -> Unit)?
    ): Notification {
        return Notification(
            type = StatusViewData.Type.SUCCESS,
            description = R.string.acq_commonsheet_paid_title,
            button = R.string.acq_commonsheet_clear_primarybutton,
            amount = amount,
            onClick = onClick
        )
    }
}
