package ru.tinkoff.acquiring.sdk.viewmodel

import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.options.screen.AttachCardOptions
import ru.tinkoff.acquiring.sdk.models.result.AsdkResult
import ru.tinkoff.acquiring.sdk.models.result.CardResult
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.toggles.FeatureToggleManager
import ru.tinkoff.acquiring.sdk.toggles.toggles.PaymentByCardV2FeatureToggle
import ru.tinkoff.acquiring.sdk.ui.delegate.AppBaseChallengeResult
import ru.tinkoff.acquiring.sdk.ui.fragments.AttachCardFlowFragmentArgs
import ru.tinkoff.acquiring.sdk.utils.panSuffix

/**
 * @author s.y.biryukov
 */
internal class AttachCardFlowViewModel(
    savedStateHandle: SavedStateHandle,
    private val featureToggleManager: FeatureToggleManager,
) : ViewModel() {
    private var isResumeFirstTime: Boolean = true
    private val args = AttachCardFlowFragmentArgs
        .fromSavedStateHandle(savedStateHandle)
    private val attachCardOptions: AttachCardOptions = args
        .attachCardOptions

    private val _commandFlow = MutableSharedFlow<Command>()
    val commandFlow =_commandFlow.asSharedFlow()

    private val _stateFlow = MutableStateFlow(State())
    val stateFlow = _stateFlow

    fun handleEvent(event: AttachCardFlowEvent) {
        when (event) {
            is AttachCardFlowEvent.CardAttached -> onCardAttachSuccess(event)
            AttachCardFlowEvent.CardAttachCancelled -> onCardAttachCancelled()
            is AttachCardFlowEvent.CardAttachError -> onCardAttachFailed(event)
            is AttachCardFlowEvent.ThreeDsError -> onThreeDsFailed(event.error)
            AttachCardFlowEvent.ThreeDsCancelled -> onThreeDsCancelled()
            is AttachCardFlowEvent.ThreeDsSuccess -> onThreeDsSuccess(event)
            is AttachCardFlowEvent.RequireThreeDs -> onRequireThreeDs(event)
            AttachCardFlowEvent.Resume -> onResume()
            is AttachCardFlowEvent.ChallengeResult -> processChallengeResult(event.data, event.card)
            is AttachCardFlowEvent.TimeOut -> showErrorAndReturnToNewCardForm(event.error, null)
        }
    }

    private fun processChallengeResult(event: AppBaseChallengeResult, card: CardResult) {
        when(event) {
            AppBaseChallengeResult.Cancelled -> onThreeDsCancelled()
            is AppBaseChallengeResult.Error -> onThreeDsFailed(event.error)
            is AppBaseChallengeResult.ProtocolError -> onThreeDsFailed(event.error)
            is AppBaseChallengeResult.RuntimeError -> onThreeDsFailed(event.error)
            is AppBaseChallengeResult.Success -> onThreeDsSuccess(card)
            is AppBaseChallengeResult.TimeOut -> Unit
            AppBaseChallengeResult.Loading -> Unit
        }
    }

    private fun onResume() {
        if (isResumeFirstTime) {
            isResumeFirstTime = false
            viewModelScope.launch {
                openAttachCardScreen()
            }
        }
    }

    private suspend fun openAttachCardScreen() {
        _commandFlow.emit(
            Command.OpenAttachCardScreen(
                attachCardOptions,
                showError = args.showError,
                isNewCardViewToggleEnabled = featureToggleManager.isEnabled(
                    PaymentByCardV2FeatureToggle
                )
            )
        )
    }

    private fun onRequireThreeDs(event: AttachCardFlowEvent.RequireThreeDs) {
        viewModelScope.launch {
            _commandFlow.emit(Command.RequireThreeDs(
                threeDsData = event.data,
                panSuffix = event.panSuffix,
                attachCardOptions = attachCardOptions
            ))
        }
    }

    private fun onThreeDsSuccess(event: AttachCardFlowEvent.ThreeDsSuccess) {
        val attachedCard = event.result as CardResult
        sendCardAttached(attachedCard)
    }

    private fun onThreeDsSuccess(attachedCard: CardResult) {
        sendCardAttached(attachedCard)
    }

    private fun onThreeDsCancelled() {
        sendCancelled()
    }

    private fun onThreeDsFailed(error: Throwable) {
        showErrorAndReturnToNewCardForm(error)
    }

    private fun onCardAttachFailed(event: AttachCardFlowEvent.CardAttachError) {
        showErrorAndReturnToNewCardForm(
            error = event.error,
            panSuffix = event.panSuffix
        )
    }

    private fun showErrorAndReturnToNewCardForm(error: Throwable, panSuffix: String? = null) {
        var message = R.string.acq_cardlist_snackbar_add_error_nopan
        var args = emptyList<Any>()
        if (panSuffix != null) {
            message = R.string.acq_cardlist_snackbar_add_error
            args = listOf(panSuffix.panSuffix())
        }
        viewModelScope.launch {
            if (featureToggleManager.isEnabled(PaymentByCardV2FeatureToggle)) {
                _commandFlow.emit(
                    Command.ShowError(
                        message = message,
                        args = args
                    )
                )
                _commandFlow.emit(
                    Command.ReturnToAttachCardScreen
                )
            } else {
                _commandFlow.emit(
                    Command.FinishWithError(
                        error = error,
                        panSuffix = panSuffix
                    )
                )
            }
        }
    }

    private fun onCardAttachCancelled() {
        sendCancelled()
    }

    private fun sendCancelled() {
        viewModelScope.launch(Dispatchers.Main) {
            _commandFlow.emit(Command.FinishWithCancel)
        }
    }

    private fun onCardAttachSuccess(event: AttachCardFlowEvent.CardAttached) {
        val attachedCard = event.card
        sendCardAttached(attachedCard)
    }

    private fun sendCardAttached(attachedCard: CardResult) {
        viewModelScope.launch {
            try {
                _commandFlow.emit(
                    Command.FinishWithAttachedCard(
                        panSuffix = requireNotNull(attachedCard.panSuffix) { "panSuffix is null" },
                        cardId = requireNotNull(attachedCard.cardId) { "cardId is null" }
                    )
                )
            } catch (e:Exception) {
                showErrorAndReturnToNewCardForm(
                    error = e,
                    panSuffix = attachedCard.panSuffix
                )
            }
        }
    }

    data class State(
        val notification: Notification? = null
    )

    sealed interface Command {
        data class OpenAttachCardScreen(
            val options: AttachCardOptions,
            val showError: Boolean,
            val isNewCardViewToggleEnabled: Boolean
        ) : Command

        data class RequireThreeDs(
            val threeDsData: ThreeDsData,
            val panSuffix: String,
            val attachCardOptions: AttachCardOptions,
        ) : Command

        data class FinishWithError(
            val error: Throwable,
            val panSuffix: String?,
        ) : Command

        data class ShowError(
            val message: Int,
            val args: List<Any> = emptyList(),
        ) : Command

        data class FinishWithAttachedCard(
            val cardId: String,
            val panSuffix: String,
        ) : Command

        data object FinishWithCancel : Command

        data object ReturnToAttachCardScreen : Command
    }
}

sealed interface AttachCardFlowEvent {
    data object Resume : AttachCardFlowEvent
    class TimeOut(val error: Throwable, val panSuffix: String?) : AttachCardFlowEvent
    class CardAttached(val card: CardResult) : AttachCardFlowEvent
    class CardAttachError(val error: Throwable, val panSuffix: String?) : AttachCardFlowEvent
    class RequireThreeDs(val data: ThreeDsData, val panSuffix: String) : AttachCardFlowEvent
    class ChallengeResult(val data: AppBaseChallengeResult, val card: CardResult) : AttachCardFlowEvent
    data object CardAttachCancelled : AttachCardFlowEvent

    class ThreeDsSuccess(val result: AsdkResult) : AttachCardFlowEvent
    class ThreeDsError(val error: Throwable, val paymentId: Long?) : AttachCardFlowEvent
    data object ThreeDsCancelled : AttachCardFlowEvent
}
