/*
 * Copyright © 2020 Tinkoff Bank
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package ru.tinkoff.acquiring.sdk.viewmodel

import android.app.Application
import android.os.Build
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.BuildConfig
import ru.tinkoff.acquiring.sdk.R
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringSdkException
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.enums.CheckType
import ru.tinkoff.acquiring.sdk.models.enums.ResponseStatus
import ru.tinkoff.acquiring.sdk.models.options.screen.AttachCardOptions
import ru.tinkoff.acquiring.sdk.models.paysources.CardData
import ru.tinkoff.acquiring.sdk.models.result.CardResult
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.PaymentByCardViewModel.Command
import ru.tinkoff.acquiring.sdk.responses.AttachCardResponse
import ru.tinkoff.acquiring.sdk.responses.Check3dsVersionResponse
import ru.tinkoff.acquiring.sdk.threeds.AppBaseProcessResult
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsAppBaseProcess
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsAppBasedTransaction
import ru.tinkoff.acquiring.sdk.threeds.ThreeDsHelper
import ru.tinkoff.acquiring.sdk.toggles.FeatureToggleManager
import ru.tinkoff.acquiring.sdk.toggles.toggles.AppBaseFeatureToggle
import ru.tinkoff.acquiring.sdk.toggles.withToggleInfo
import ru.tinkoff.acquiring.sdk.ui.customview.status.StatusViewData
import ru.tinkoff.acquiring.sdk.ui.fragments.AttachCardFragmentArgs
import ru.tinkoff.acquiring.sdk.utils.NotificationFactory
import ru.tinkoff.acquiring.sdk.utils.panSuffix

/**
 * @author Mariya Chernyadieva
 */
internal class AttachCardViewModel(
    application: Application,
    savedStateHandle: SavedStateHandle,
    private val dsAppBaseProcess: ThreeDsAppBaseProcess,
    private val toggleManager: FeatureToggleManager
) : AndroidViewModel(application) {

    private var firstTimeResume: Boolean = true
    private var cardData: CardData? = null
    private var isCardDataValid: Boolean = false
    private var args: AttachCardFragmentArgs =
        AttachCardFragmentArgs.fromSavedStateHandle(savedStateHandle)
    private val options: AttachCardOptions = args.attachCardOptions

    private val sdk = AcquiringSdk(
        publicKey = options.publicKey,
        terminalKey = options.terminalKey,
    )
    private val _stateFlow = MutableStateFlow(
        State(
            options = options
        )
    )
    val stateFlow = _stateFlow.asStateFlow()
    private val _commandFlow = MutableSharedFlow<Command>()
    val commandFlow = _commandFlow.asSharedFlow()

    fun handleEvent(event: Event) {
        when (event) {
            Event.AttachButtonClicked -> onAttachButtonClicked()
            Event.BackButtonClicked -> onBackButtonClicked()
            is Event.CardDataChanged -> onCardDataChanged(
                cardData = event.cardData,
                isValid = event.isValid,
            )

            Event.ShowChallengeNotifiaction -> {
                _stateFlow.update {
                    it.copy(
                        notification = Notification(
                            type = StatusViewData.Type.PROGRESS,
                            title = R.string.acq_commonsheet_processing_title,
                            description = R.string.acq_commonsheet_processing_description,
                        )
                    )
                }
            }

            is Event.ChallengeTimeoutNotification -> {
                val onClick: () -> Unit = {
                    viewModelScope.launch {
                        _commandFlow.emit(
                            Command.FinishWithTimeOut(
                                error = event.error,
                                panSuffix = cardData?.pan,
                            )
                        )
                    }
                }
                _stateFlow.update {
                    val notification = NotificationFactory.getPaymentTimeoutErrorNotification(
                        exception = event.error,
                        onClickFactory = { onClick }
                    )
                    it.copy(
                        notification = notification
                    )
                }
            }

            is Event.RequestGetCardState -> requestAddCardState(requestKey = event.requestKey)

            Event.Resume -> onViewResume()
        }
    }

    private fun onViewResume() {
        if (firstTimeResume) {
            onViewResumeFirstTime()
        }
    }

    private fun onViewResumeFirstTime() {
        firstTimeResume = false
        viewModelScope.launch {
            _commandFlow.emit(Command.RequestCardFieldFocus)
        }
    }

    private fun requestAddCardState(requestKey: String?) {
        viewModelScope.launch(Dispatchers.IO) {
            sdk.getAddCardState {
                this.requestKey = requestKey
            }.execute(onSuccess = { response ->
                if (response.status == ResponseStatus.COMPLETED) {
                    viewModelScope.launch {
                        _commandFlow.emit(
                            Command.FinishWithSuccess(
                                CardResult(
                                    cardId = response.cardId, panSuffix = cardData?.pan?.panSuffix()
                                )
                            )
                        )
                    }
                } else {
                    val throwable = AcquiringSdkException(
                        IllegalStateException("AsdkState = ${response.status}"),
                        errorCode = response.errorCode,
                    )
                    viewModelScope.launch {
                        _commandFlow.emit(
                            Command.FinishWithError(
                                error = throwable,
                                panSuffix = cardData?.pan?.panSuffix()
                            )
                        )
                    }
                }
            }, onFailure = {
                viewModelScope.launch {
                    _commandFlow.emit(
                        Command.FinishWithError(
                            error = it,
                            panSuffix = cardData?.pan?.panSuffix()
                        )
                    )
                }
            })
        }
    }


    private fun onAttachButtonClicked() {
        val attachCardOptions = options
        val customerOptions = attachCardOptions.customer
        val customerKey = requireNotNull(customerOptions.customerKey) { "Не указан customerKey" }
        val checkType = attachCardOptions.customer.checkType
        val data = customerOptions.data
        val cData = requireNotNull(cardData) { "Нет информации по карте" }
        startAttachCard(cData, customerKey, checkType, data)
    }

    private fun onBackButtonClicked() {
        sendCancel()
    }

    private fun onCardDataChanged(cardData: CardData, isValid: Boolean) {
        this.cardData = cardData
        this.isCardDataValid = isValid
    }

    private fun attachCard(
        requestKey: String,
        data: Map<String, String>?,
        check3dsVersionResponse: Check3dsVersionResponse? = null,
        transaction: ThreeDsAppBasedTransaction? = null,
        paymentId: Long? = null
    ) {
        viewModelScope.launch(Dispatchers.IO) {
            val dataWithToggleInfo = data.withToggleInfo()
            val attachCardRequest = sdk.attachCard {
                this.requestKey = requestKey
                this.data = dataWithToggleInfo
                this.cardData = this@AttachCardViewModel.cardData
                if (check3dsVersionResponse?.is3DsVersionV2() == true) {
                    this.addContentHeader()
                    this.addUserAgentHeader()
                }
                this.sdkVersion = BuildConfig.ASDK_VERSION_NAME
                this.softwareVersion = Build.VERSION.SDK_INT.toString()
                this.deviceModel = Build.MODEL
            }
            try {
                val result = attachCardRequest.execute()
                handleAttachSuccess(result, check3dsVersionResponse, transaction, paymentId)
            } catch (e: Exception) {
                handleAttachError(e)
            }
        }
    }

    private fun handleAttachSuccess(
        it: AttachCardResponse,
        check3dsVersionResponse: Check3dsVersionResponse?,
        transaction: ThreeDsAppBasedTransaction? = null,
        paymentId: Long? = null,
    ) {
        when (it.status) {
            ResponseStatus.THREE_DS_CHECKING -> {
                viewModelScope.launch {
                    if (check3dsVersionResponse?.isAppBase() == true
                        && toggleManager.isEnabled(AppBaseFeatureToggle)
                    ) {
                        _commandFlow.emit(
                            Command.Start3DsChallenge(
                                threeDsData = ThreeDsData(
                                    acsUrl = it.acsUrl,
                                    paymentId = paymentId,
                                    requestKey = it.requestKey
                                ).apply {
                                    acsRefNumber = it.acsReferenceNumber
                                    acsTransId = it.acsTransId
                                    acsSignedContent = it.acsSignedContent
                                    tdsServerTransId =
                                        checkNotNull(check3dsVersionResponse.serverTransId)
                                },
                                transaction = requireNotNull(transaction),
                                options = options,
                                cardResult = CardResult(
                                    it.cardId,
                                    cardData?.pan?.panSuffix()
                                )
                            )
                        )
                    } else {
                        val threeDsData =
                            it.getThreeDsData(paymentSys = check3dsVersionResponse?.paymentSystem)
                        if (check3dsVersionResponse?.is3DsVersionV2() == true) {
                            ThreeDsHelper.CollectData.addExtraThreeDsData(
                                data = threeDsData,
                                acsTransId = checkNotNull(it.acsTransId),
                                serverTransId = checkNotNull(check3dsVersionResponse.serverTransId),
                                version = checkNotNull(check3dsVersionResponse.version)
                            )
                        }
                        _commandFlow.emit(
                            Command.Open3DSScreen(
                                data = threeDsData,
                                panSuffix = cardData?.pan?.panSuffix()!!,
                                options = options,
                            )
                        )
                    }
                }
            }

            null -> {
                viewModelScope.launch {
                    _commandFlow.emit(
                        Command.FinishWithSuccess(
                            cardResult = CardResult(it.cardId, cardData?.pan?.panSuffix())
                        )
                    )
                }
            }

            else -> {
                val e = AcquiringSdkException(
                    IllegalStateException("ResponseStatus = ${it.status}")
                )
                handleAttachError(e)
            }
        }
    }

    private fun emitError(exception: Exception) {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.FinishWithError(
                    error = exception,
                    panSuffix = cardData?.pan
                )
            )
        }
    }

    private fun handleAttachError(e: Exception) {
        AcquiringSdk.log(e)
        if (args.showError) {
            _stateFlow.update {
                it.copy(
                    isLoading = false,
                    notification = Notification(
                        type = StatusViewData.Type.ERROR,
                        title = R.string.acq_attach_card_error,
                        description = R.string.acq_generic_stub_description,
                        button = R.string.acq_generic_alert_access,
                        onClick = {
                            emitError(e)
                        }
                    )
                )
            }
        } else {
            emitError(e)
        }
    }

    private fun sendCancel() {
        viewModelScope.launch {
            _commandFlow.emit(Command.Cancel)
        }
    }

    private fun startAttachCard(
        cardData: CardData,
        customerKey: String,
        checkType: String,
        data: Map<String, String>?
    ) {
        this.cardData = cardData

        _stateFlow.update {
            it.copy(isLoading = true)
        }

        viewModelScope.launch(Dispatchers.IO) {
            val addCardRequest = sdk.addCard {
                this.customerKey = customerKey
                this.checkType = checkType
            }

            try {
                val it = addCardRequest.execute()
                check3dsVersionIfNeed(cardData, it.paymentId, it.requestKey!!, data)
            } catch (e: Exception) {
                AcquiringSdk.log(e)
                handleAttachError(e)
            }
        }
    }

    private fun check3dsVersionIfNeed(
        cardData: CardData,
        paymentId: Long?,
        requestKey: String,
        data: Map<String, String>?
    ) {
        when (options.customer.checkType) {
            CheckType.THREE_DS.toString(),
            CheckType.THREE_DS_HOLD.toString(),
            CheckType.AUTO -> {
                viewModelScope.launch(Dispatchers.IO) {
                    try {
                        val check3DsRequest = sdk.check3DsVersion {
                            this.paymentId = paymentId
                            this.paymentSource = cardData
                        }
                        val result = check3DsRequest.execute()

                        add3dsVersionToContext(result)

                        when {
                            result.isAppBase() && toggleManager.isEnabled(AppBaseFeatureToggle) -> {
                                when (val appBaseResult = dsAppBaseProcess.collectInfo(result)) {

                                    is AppBaseProcessResult.Error -> {
                                        handleAttachError(appBaseResult.throwable)
                                    }

                                    is AppBaseProcessResult.Success -> {
                                        attachCard(
                                            requestKey,
                                            appBaseResult.threedsData + (data ?: mapOf()),
                                            result,
                                            appBaseResult.transaction,
                                            paymentId
                                        )
                                    }
                                }
                            }

                            result.is3DsVersionV2() -> {
                                val check3dsMap: MutableMap<String, String> = get3DsParams(result)
                                attachCard(
                                    requestKey,
                                    check3dsMap + (data ?: mapOf()),
                                    result,
                                )
                            }

                            else -> {
                                attachCard(requestKey, data, result)
                            }
                        }
                    } catch (e: Exception) {
                        handleAttachError(e)
                    }
                }
            }

            else -> {
                attachCard(requestKey, data)
            }
        }
    }

    private fun add3dsVersionToContext(result: Check3dsVersionResponse) {
        try {
            val threeDsVersion = when {
                result.isAppBase() -> FeatureToggleManager.FeatureToggleContext.VERSION_2_APP
                result.is3DsVersionV2() -> FeatureToggleManager.FeatureToggleContext.VERSION_2_BROWSER
                else -> FeatureToggleManager.FeatureToggleContext.VERSION_1
            }
            viewModelScope.launch {
                toggleManager.updateFeatureToggleContext {
                    it.copy(
                        threeDsVersion = threeDsVersion,
                        appVersion = AcquiringSdk.appVersion,
                    )
                }
            }
        } catch (e: Exception) {
            AcquiringSdk.log(e)
        }
    }

    private suspend fun get3DsParams(result: Check3dsVersionResponse): MutableMap<String, String> {
        var check3dsMap: MutableMap<String, String>
        withContext(Dispatchers.Main) {
            check3dsMap = ThreeDsHelper.CollectData.invoke(getApplication(), result)
            ThreeDsHelper.CollectData.addExtraData(check3dsMap, result)
        }
        return check3dsMap
    }

    sealed interface Event {
        data object Resume : Event
        data object AttachButtonClicked : Event
        data object BackButtonClicked : Event
        data object ShowChallengeNotifiaction : Event
        class ChallengeTimeoutNotification(
            val error: Throwable
        ) : Event

        class RequestGetCardState(
            val requestKey: String? = null
        ) : Event

        class CardDataChanged(
            val cardData: CardData,
            val isValid: Boolean,
        ) : Event
    }

    data class State(
        val options: AttachCardOptions,
        val isLoading: Boolean = false,
        val notification: Notification? = null,
    )

    sealed interface Command {
        data class FinishWithSuccess(
            val cardResult: CardResult
        ) : Command

        data class Open3DSScreen(
            val data: ThreeDsData,
            val panSuffix: String,
            val options: AttachCardOptions,
        ) : Command

        data class FinishWithTimeOut(
            val error: Throwable,
            val panSuffix: String?,
        ) : Command

        data class Start3DsChallenge(
            val transaction: ThreeDsAppBasedTransaction,
            val threeDsData: ThreeDsData,
            val options: AttachCardOptions,
            val cardResult: CardResult
        ) : Command

        data class FinishWithError(
            val error: Exception,
            val panSuffix: String?
        ) : Command

        data object Cancel : Command

        data object RequestCardFieldFocus : Command
    }
}
