/*
 * Copyright © 2020 Tinkoff Bank
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package ru.tinkoff.acquiring.sdk.viewmodel

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.localization.ASDKString
import ru.tinkoff.acquiring.sdk.models.enums.DataTypeQr
import ru.tinkoff.acquiring.sdk.models.enums.ResponseStatus
import ru.tinkoff.acquiring.sdk.models.options.screen.PaymentOptions
import ru.tinkoff.acquiring.sdk.payment.methods.InitConfigurator.configure
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.ui.customview.status.StatusViewData

internal class DynamicQrCodeViewModel(
    private val sdk: AcquiringSdk,
    private val paymentOptions: PaymentOptions,
) : ViewModel() {


    private var isResumeFirstTime: Boolean = true
    private val _stateFlow = MutableStateFlow(
        State(
            paymentOptions = paymentOptions
        )
    )
    val stateFlow = _stateFlow.asStateFlow()

    private val _commandFlow = MutableSharedFlow<Command>()
    val commandFlow = _commandFlow.asSharedFlow()
    fun onErrorButtonClick() {
        loadQr()
    }

    private fun loadQr() {
        getDynamicQr(paymentOptions)
    }

    private fun getDynamicQr(paymentOptions: PaymentOptions) {
        _stateFlow.update {
            it.copy(
                isLoading = true
            )
        }
        viewModelScope.launch(Dispatchers.IO) {
            val init = sdk.init { configure(paymentOptions) }
            init.execute(
                onSuccess = {
                    val paymentId = requireNotNull(it.paymentId) { "it.paymentId is null" }
                    getQr(paymentId, DataTypeQr.IMAGE)
                },
                onFailure = {
                    showError(it, paymentOptions.paymentId)
                }
            )
        }
    }

    private fun getDynamicQrLink(paymentOptions: PaymentOptions) {
        _stateFlow.update {
            it.copy(isLoading = true)
        }
        viewModelScope.launch(Dispatchers.IO) {
            val init = sdk.init { configure(paymentOptions) }
            init.execute(
                onSuccess = {
                    val paymentId = requireNotNull(it.paymentId) { "it.paymentId is null" }
                    getQr(paymentId, DataTypeQr.PAYLOAD)
                },
                onFailure = {
                    showError(it, paymentOptions.paymentId)
                }
            )
        }
    }

    private fun getQr(paymentId: Long, type: DataTypeQr) {
        _stateFlow.update {
            it.copy(
                isLoading = true,
                showProgressDialog = type == DataTypeQr.PAYLOAD
            )
        }
        viewModelScope.launch(Dispatchers.IO) {
            val request = sdk.getQr {
                this.paymentId = paymentId
                dataType = type
            }

            request.execute(
                onSuccess = {
                    val data = it.data!!
                    when (type) {
                        DataTypeQr.IMAGE -> {
                            _stateFlow.update {
                                it.copy(
                                    isLoading = false,
                                    qrImage = data
                                )
                            }
                            viewModelScope.launch {
                                delay(1500)
                                getState(paymentId)
                            }
                        }

                        DataTypeQr.PAYLOAD -> {
                            _stateFlow.update {
                                it.copy(
                                    isLoading = false,
                                    showProgressDialog = false
                                )
                            }

                            viewModelScope.launch {
                                _commandFlow.emit(Command.OpenLink(data))
                            }
                        }
                    }
                },
                onFailure = {
                    showError(it, paymentId)
                }
            )
        }
    }

    private fun getState(paymentId: Long) {
        viewModelScope.launch(Dispatchers.IO) {

            val request = sdk.getState {
                this.paymentId = paymentId
            }

            request.execute(
                onSuccess = { response ->
                    if (response.status == ResponseStatus.CONFIRMED || response.status == ResponseStatus.AUTHORIZED) {
                        viewModelScope.launch {
                            _commandFlow.emit(
                                Command.ReturnSuccessResult(
                                    paymentId = paymentId
                                )
                            )
                        }
                    } else {
                        viewModelScope.launch {
                            delay(5000)
                            getState(paymentId)
                        }
                    }
                },
                onFailure = {
                    showError(it, paymentId)
                }
            )
        }
    }

    fun handleEvent(event: Event) {
        when (event) {
            Event.Resume -> onResume()
            Event.ShareButtonClick -> onShareButtonClick()
            Event.BackPressed -> onBackPressed()
        }
    }

    private fun onBackPressed() {
        viewModelScope.launch {
            _commandFlow.emit(Command.ReturnCancelResult)
        }
    }

    private fun onShareButtonClick() {
        getDynamicQrLink(paymentOptions)
    }

    private fun onResume() {
        if (isResumeFirstTime) {
            isResumeFirstTime = false
            onResumeFirstTime()
        }
    }

    private fun onResumeFirstTime() {
        getDynamicQr(paymentOptions)
    }

    private fun showError(error: Exception, paymentId: Long?) {
        _stateFlow.update {
            it.copy(
                showProgressDialog = false,
                isLoading = false,
                error = error,
                notification = Notification(
                    type = StatusViewData.Type.ERROR,
                    description = ASDKString.acq_pay_dialog_error_fallback_message,
                    button = ASDKString.acq_generic_alert_access,
                    onClick = {
                        finishWithError(error, paymentId)
                    }
                )
            )
        }
    }

    private fun finishWithError(error: Exception, paymentId: Long? = null) {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.ReturnErrorResult(
                    error = error,
                    paymentId = paymentId
                )
            )
        }
    }

    data class State(
        val paymentOptions: PaymentOptions,
        val isLoading: Boolean = false,
        val showProgressDialog: Boolean = false,
        val qrImage: String? = null,
        val error: Throwable? = null,
        val notification: Notification? = null
    )

    sealed interface Command {
        class OpenLink(val url: String) : Command
        class ReturnSuccessResult(val paymentId: Long) : Command
        class ReturnErrorResult(val paymentId: Long? = null, val error: Throwable) : Command
        data object ReturnCancelResult : Command
    }

    sealed interface Event {
        data object Resume : Event
        data object ShareButtonClick : Event
        data object BackPressed : Event
    }
}
