/*
 * Copyright © 2020 Tinkoff Bank
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package ru.tinkoff.acquiring.sdk.viewmodel

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.localization.ASDKString
import ru.tinkoff.acquiring.sdk.models.enums.DataTypeQr
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.ui.customview.status.StatusViewData

internal class StaticQrCodeViewModel(
    private val sdk: AcquiringSdk,
) : ViewModel() {

    private var isResumeFirstTime: Boolean = true
    private val _stateFlow = MutableStateFlow(State())
    val stateFlow = _stateFlow.asStateFlow()

    private val _commandFlow = MutableSharedFlow<Command>()
    val commandFlow = _commandFlow.asSharedFlow()

    private fun getStaticQr() {
        _stateFlow.update {
            it.copy(isLoading = true)
        }

        viewModelScope.launch(Dispatchers.IO) {
            val request = sdk.getStaticQr {
                data = DataTypeQr.IMAGE
            }

            request.execute(
                onSuccess = { response ->
                    val data = requireNotNull(response.data) { "response.data is null" }
                    _stateFlow.update {
                        it.copy(
                            isLoading = false,
                            qrImage = data
                        )
                    }
                },
                onFailure = ::showError
            )
        }
    }

    private fun onShareButtonClicked() {
        viewModelScope.launch(Dispatchers.IO) {
            _stateFlow.update {
                it.copy(
                    isLoading = true,
                    showProgressDialog = false
                )
            }

            val request = sdk.getStaticQr {
                data = DataTypeQr.PAYLOAD
            }

            request.execute(
                onSuccess = { response ->
                    _stateFlow.update {
                        it.copy(
                            showProgressDialog = false,
                            isLoading = false
                        )
                    }
                    val url = response.data
                    url?.let {
                        viewModelScope.launch {
                            _commandFlow.emit(Command.OpenLink(url))
                        }
                    }
                },
                onFailure = ::showError
            )
        }
    }

    private fun showError(error: Exception) {
        _stateFlow.update {
            it.copy(
                showProgressDialog = false,
                isLoading = false,
                error = error,
                notification = Notification(
                    type = StatusViewData.Type.ERROR,
                    description = ASDKString.acq_pay_dialog_error_fallback_message,
                    button = ASDKString.acq_generic_alert_access,
                    onClick = {
                        finishWithError(error)
                    }
                )
            )
        }
    }

    private fun finishWithError(error: Exception) {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.ReturnErrorResult(
                    error = error
                )
            )
        }
    }

    fun handleEvent(event: Event) {
        when (event) {
            Event.Resume -> onResume()
            Event.ShareButtonClick -> onShareButtonClicked()
            Event.BackPressed -> onBackPressed()
        }
    }

    private fun onBackPressed() {
        viewModelScope.launch {
            _commandFlow.emit(Command.ReturnCancelResult)
        }
    }

    private fun onResume() {
        if (isResumeFirstTime) {
            isResumeFirstTime = false
            onResumeFirstTime()
        }
    }

    private fun onResumeFirstTime() {
        getStaticQr()
    }

    data class State(
        val isLoading: Boolean = false,
        val showProgressDialog: Boolean = false,
        val qrImage: String? = null,
        val error: Throwable? = null,
        val notification: Notification? = null,
    )

    sealed interface Command {
        class OpenLink(val url: String) : Command
        class ReturnErrorResult(val error: Throwable) : Command
        data object ReturnCancelResult : Command
        class ReturnSuccessResult(
            val paymentId: Long
        ) : Command
    }

    sealed interface Event {
        data object Resume : Event
        data object BackPressed : Event
        data object ShareButtonClick : Event
    }
}
