/*
 * Copyright © 2020 Tinkoff Bank
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package ru.tinkoff.acquiring.sdk.viewmodel

import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.flowOn
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import org.json.JSONObject
import ru.tinkoff.acquiring.sdk.AcquiringSdk
import ru.tinkoff.acquiring.sdk.exceptions.AcquiringSdkException
import ru.tinkoff.acquiring.sdk.models.ThreeDsData
import ru.tinkoff.acquiring.sdk.models.enums.ResponseStatus
import ru.tinkoff.acquiring.sdk.models.result.CardResult
import ru.tinkoff.acquiring.sdk.models.result.PaymentResult
import ru.tinkoff.acquiring.sdk.network.AcquiringApi
import ru.tinkoff.acquiring.sdk.payment.pooling.GetStatusMethod
import ru.tinkoff.acquiring.sdk.payment.pooling.GetStatusPooling
import ru.tinkoff.acquiring.sdk.redesign.payment.ui.Notification
import ru.tinkoff.acquiring.sdk.ui.activities.ThreeDsFragmentArgs
import ru.tinkoff.acquiring.sdk.utils.Base64
import java.net.URLEncoder

internal class ThreeDsViewModel(
    savedStateHandle: SavedStateHandle,
    private val sdk: AcquiringSdk,
) : ViewModel() {

    private var isFirstTimeResume: Boolean = true
    private val args: ThreeDsFragmentArgs =
        ThreeDsFragmentArgs.fromSavedStateHandle(savedStateHandle)

    private val cancelActions = arrayOf("cancel.do", "cancel=true")
    private var termUrl: String? = null
    private var isCancelled: Boolean = false
    private val getStatusPooling = GetStatusPooling(
        GetStatusMethod.Impl(sdk))
    private var requestPaymentStateJob: Job? = null

    private val _stateFlow = MutableStateFlow(State())
    val stateFlow = _stateFlow.asStateFlow()

    private val _commandFlow = MutableSharedFlow<Command>()
    val commandFlow = _commandFlow

    fun handleEvent(event: Event) {
        when (event) {
            is Event.WebPageLoaded -> {
                setProgress(false)

                if (cancelActions.any { event.url.contains(it) }) {
                    onCancelPageLoaded()
                }

                if (!isCancelled && termUrl == event.url) {
                    onDataPageLoaded()
                }
            }

            Event.ViewResume -> {
                if (isFirstTimeResume) {
                    isFirstTimeResume = false
                    startThreeDs()
                }
            }

            is Event.WebPageLoadingError -> {
                viewModelScope.launch {
                    val message = event.description?.toString() ?: "Code: ${event.errorCode}"
                    _commandFlow.emit(
                        Command.ReturnError(
                            error = IllegalStateException(message),
                            paymentId = args.threeDsData.paymentId
                        )
                    )
                }
            }
        }
    }

    private fun onDataPageLoaded() {
        _stateFlow.update {
            it.copy(isWebViewInvisible = true)
        }
        requestState(args.threeDsData)
    }

    private fun onCancelPageLoaded() {
        isCancelled = true
        viewModelScope.launch {
            _commandFlow.emit(Command.ReturnCancel)
        }
    }

    private fun startThreeDs() {
        val data = args.threeDsData
        val url = requireNotNull(data.acsUrl) { "acsUrl is null" }
        val params: String?


        if (data.is3DsVersion2) {
            termUrl = TERM_URL_V2
            val base64Creq = prepareCreqParams()
            val creq = URLEncoder.encode(base64Creq, UTF8)
            params = "creq=$creq"
        } else {
            termUrl = TERM_URL
            val paReq = URLEncoder.encode(data.paReq, UTF8)
            val md = URLEncoder.encode(data.md, UTF8)
            val termUrl = URLEncoder.encode(termUrl, UTF8)
            params = "PaReq=$paReq&MD=$md&TermUrl=$termUrl"
        }

        viewModelScope.launch {
            setProgress(true)
            _commandFlow.emit(
                Command.StartThreeDs(
                    url = url,
                    params = params
                )
            )
        }
    }

    private fun setProgress(isVisible: Boolean) {
        _stateFlow.update {
            it.copy(
                inProgress = isVisible
            )
        }
    }

    private fun prepareCreqParams(): String {
        val data = args.threeDsData
        val creqData = JSONObject().apply {
            put("threeDSServerTransID", data.tdsServerTransId)
            put("acsTransID", data.acsTransId)
            put("messageVersion", data.version)
            put("challengeWindowSize", WINDOW_SIZE_CODE)
            put("messageType", MESSAGE_TYPE)
        }
        return Base64.encodeToString(
            creqData.toString().toByteArray(), Base64.NO_PADDING or Base64.NO_WRAP
        ).trim()
    }

    private fun requestState(threeDsData: ThreeDsData) {
        if (threeDsData.isPayment) {
            val value = threeDsData.paymentId
            requestPaymentState(requireNotNull(value) { "threeDsData.paymentId is null" })
        } else if (threeDsData.isAttaching) {
            requestAddCardState(threeDsData.requestKey)
        }
    }

    private fun requestPaymentState(paymentId: Long) {
        requestPaymentStateJob?.cancel()
        requestPaymentStateJob = viewModelScope.launch {
            setProgress(true)
            getStatusPooling.start(paymentId = paymentId).flowOn(Dispatchers.IO)
                .catch { handleException(it, paymentId) }
                .filter { ResponseStatus.checkSuccessStatuses(it) }
                .collect { handleConfirmOnAuthStatus(paymentId) }
        }
    }

    private fun requestAddCardState(requestKey: String?) {
        setProgress(true)
        viewModelScope.launch(Dispatchers.IO) {
            sdk.getAddCardState {
                this.requestKey = requestKey
            }.execute(onSuccess = { response ->
                setProgress(false)
                if (response.status == ResponseStatus.COMPLETED) {
                    returnAddCardResult(
                        CardResult(
                            cardId = response.cardId, panSuffix = args.panSuffix
                        )
                    )
                } else {
                    val throwable = AcquiringSdkException(
                        IllegalStateException("AsdkState = ${response.status}"),
                        errorCode = response.errorCode,
                    )
                    handleException(throwable)
                }
            }, onFailure = {
                setProgress(false)
                handleException(it)
            })
        }
    }

    private fun returnAddCardResult(cardResult: CardResult) {
        viewModelScope.launch {
            _commandFlow.emit(
                Command.ReturnAddCardSuccessResult(cardResult)
            )
        }
    }

    private fun handleConfirmOnAuthStatus(paymentId: Long) {
        setProgress(false)
        returnPaymentResult(PaymentResult(paymentId))
    }

    private fun returnPaymentResult(paymentResult: PaymentResult) {
        viewModelScope.launch {
            _commandFlow.emit(Command.ReturnPaymentSuccessResult(paymentResult))
        }
    }

    private fun handleException(throwable: Throwable, paymentId: Long? = null) {
        //TODO Доработать парсинг ошибок и разбор по типам
        viewModelScope.launch {
            _commandFlow.emit(
                Command.ReturnError(
                    error = throwable,
                    paymentId = paymentId
                )
            )
        }
    }

    data class State(
        val isWebViewInvisible: Boolean = false,
        val notification: Notification? = null,
        val inProgress: Boolean = false,
    )

    sealed interface Event {
        data class WebPageLoaded(
            val url: String
        ) : Event

        class WebPageLoadingError(
            val errorCode: Int, val description: CharSequence?
        ) : Event

        data object ViewResume : Event
    }

    sealed interface Command {
        data class StartThreeDs(
            val url: String, val params: String
        ) : Command

        data class ReturnPaymentSuccessResult(
            val paymentResult: PaymentResult
        ) : Command

        data class ReturnAddCardSuccessResult(val cardResult: CardResult) : Command
        data class ReturnError(val error: Throwable, val paymentId: Long?) : Command
        data object ReturnCancel : Command
    }

    companion object {
        private const val MESSAGE_TYPE = "CReq"
        private const val WINDOW_SIZE_CODE = "05"
        private const val UTF8 = "UTF-8"
        val TERM_URL
            get() = "${AcquiringApi.getUrl()}/${AcquiringApi.SUBMIT_3DS_AUTHORIZATION}"
        val TERM_URL_V2
            get() = "${AcquiringApi.getUrl()}/${AcquiringApi.SUBMIT_3DS_AUTHORIZATION_V2}"
    }
}
