/*
 * Decompiled with CFR 0.152.
 */
package ru.tinkoff.kora.resilient.circuitbreaker;

import jakarta.annotation.Nonnull;
import jakarta.annotation.Nullable;
import java.time.Clock;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.tinkoff.kora.common.util.TimeUtils;
import ru.tinkoff.kora.resilient.circuitbreaker.CallNotPermittedException;
import ru.tinkoff.kora.resilient.circuitbreaker.CircuitBreaker;
import ru.tinkoff.kora.resilient.circuitbreaker.CircuitBreakerConfig;
import ru.tinkoff.kora.resilient.circuitbreaker.CircuitBreakerMetrics;
import ru.tinkoff.kora.resilient.circuitbreaker.CircuitBreakerPredicate;

final class KoraCircuitBreaker
implements CircuitBreaker {
    private static final Logger logger = LoggerFactory.getLogger(KoraCircuitBreaker.class);
    private static final long CLOSED_COUNTER_MASK = Integer.MAX_VALUE;
    private static final long CLOSED_STATE = Long.MIN_VALUE;
    private static final long HALF_OPEN_COUNTER_MASK = 65535L;
    private static final long HALF_OPEN_STATE = 0x4000000000000000L;
    private static final long HALF_OPEN_INCREMENT_SUCCESS = 65536L;
    private static final long HALF_OPEN_INCREMENT_ERROR = 0x100000000L;
    private static final long OPEN_STATE = 0L;
    private static final long COUNTER_INC = 1L;
    private static final long ERR_COUNTER_INC = 0x80000000L;
    private static final long BOTH_COUNTERS_INC = 0x80000001L;
    private final AtomicLong state = new AtomicLong(Long.MIN_VALUE);
    private final String name;
    private final CircuitBreakerConfig.NamedConfig config;
    private final CircuitBreakerPredicate failurePredicate;
    private final CircuitBreakerMetrics metrics;
    private final long waitDurationInOpenStateInMillis;
    private final Clock clock;

    KoraCircuitBreaker(String name, CircuitBreakerConfig.NamedConfig config, CircuitBreakerPredicate failurePredicate, CircuitBreakerMetrics metrics) {
        this.name = name;
        this.config = config;
        this.failurePredicate = failurePredicate;
        this.metrics = metrics;
        this.waitDurationInOpenStateInMillis = config.waitDurationInOpenState().toMillis();
        this.clock = Clock.systemDefaultZone();
        this.metrics.recordState(name, CircuitBreaker.State.CLOSED);
    }

    @Nonnull
    CircuitBreaker.State getState() {
        if (Boolean.FALSE.equals(this.config.enabled())) {
            logger.debug("CircuitBreaker '{}' is disabled", (Object)this.name);
            return CircuitBreaker.State.CLOSED;
        }
        return this.getState(this.state.get());
    }

    @Override
    public <T> T accept(@Nonnull Supplier<T> callable) {
        return this.internalAccept(callable, null);
    }

    @Override
    public <T> T accept(@Nonnull Supplier<T> callable, @Nonnull Supplier<T> fallback) {
        return this.internalAccept(callable, fallback);
    }

    private <T> T internalAccept(@Nonnull Supplier<T> supplier, @Nullable Supplier<T> fallback) {
        if (Boolean.FALSE.equals(this.config.enabled())) {
            logger.debug("CircuitBreaker '{}' is disabled", (Object)this.name);
            this.metrics.recordCallAcquire(this.name, CircuitBreakerMetrics.CallAcquireStatus.DISABLED);
            return supplier.get();
        }
        try {
            this.acquire();
            T t = supplier.get();
            this.releaseOnSuccess();
            return t;
        }
        catch (CallNotPermittedException e) {
            if (fallback == null) {
                throw e;
            }
        }
        catch (Throwable e) {
            this.releaseOnError(e);
            throw e;
        }
        return fallback.get();
    }

    private CircuitBreaker.State getState(long value) {
        return switch ((int)(value >> 62 & 3L)) {
            case 0 -> CircuitBreaker.State.OPEN;
            case 1 -> CircuitBreaker.State.HALF_OPEN;
            default -> CircuitBreaker.State.CLOSED;
        };
    }

    private int countClosedErrors(long value) {
        return (int)(value >> 31 & Integer.MAX_VALUE);
    }

    private int countClosedTotal(long value) {
        return (int)(value & Integer.MAX_VALUE);
    }

    private short countHalfOpenSuccess(long value) {
        return (short)(value >> 16 & 0xFFFFL);
    }

    private short countHalfOpenError(long value) {
        return (short)(value >> 32 & 0xFFFFL);
    }

    private short countHalfOpenAcquired(long value) {
        return (short)(value & 0xFFFFL);
    }

    private long getOpenState() {
        return this.clock.millis();
    }

    private void onStateChange(@Nonnull CircuitBreaker.State prevState, @Nonnull CircuitBreaker.State newState, String action, @Nullable Throwable throwable) {
        if (throwable != null) {
            logger.info("CircuitBreaker '{}' switched from {} to {} on {} due to: {}", new Object[]{this.name, prevState, newState, action, throwable.toString()});
        } else {
            logger.info("CircuitBreaker '{}' switched from {} to {} on {}", new Object[]{this.name, prevState, newState, action});
        }
        this.metrics.recordState(this.name, newState);
    }

    @Override
    public void acquire() throws CallNotPermittedException {
        if (Boolean.FALSE.equals(this.config.enabled())) {
            logger.debug("CircuitBreaker '{}' is disabled", (Object)this.name);
            this.metrics.recordCallAcquire(this.name, CircuitBreakerMetrics.CallAcquireStatus.DISABLED);
            return;
        }
        if (!this.tryAcquire()) {
            throw new CallNotPermittedException(this.getState(this.state.get()), this.name);
        }
    }

    @Override
    public boolean tryAcquire() {
        if (Boolean.FALSE.equals(this.config.enabled())) {
            logger.debug("CircuitBreaker '{}' is disabled", (Object)this.name);
            this.metrics.recordCallAcquire(this.name, CircuitBreakerMetrics.CallAcquireStatus.DISABLED);
            return true;
        }
        long stateLong = this.state.get();
        CircuitBreaker.State state = this.getState(stateLong);
        if (state == CircuitBreaker.State.CLOSED) {
            logger.trace("CircuitBreaker '{}' acquired in CLOSED state", (Object)this.name);
            this.metrics.recordCallAcquire(this.name, CircuitBreakerMetrics.CallAcquireStatus.PERMITTED);
            return true;
        }
        if (state == CircuitBreaker.State.HALF_OPEN) {
            short acquired = this.countHalfOpenAcquired(stateLong);
            if (acquired < this.config.permittedCallsInHalfOpenState()) {
                boolean isAcquired = this.state.compareAndSet(stateLong, stateLong + 1L);
                if (isAcquired) {
                    if (logger.isTraceEnabled()) {
                        logger.trace("CircuitBreaker '{}' acquired in HALF_OPEN state with {} calls left", (Object)this.name, (Object)(acquired - 1));
                    }
                    this.metrics.recordCallAcquire(this.name, CircuitBreakerMetrics.CallAcquireStatus.PERMITTED);
                    return true;
                }
                return this.tryAcquire();
            }
            logger.trace("CircuitBreaker '{}' rejected in HALF_OPEN state due to all {} calls acquired", (Object)this.name, (Object)acquired);
            this.metrics.recordCallAcquire(this.name, CircuitBreakerMetrics.CallAcquireStatus.REJECTED);
            return false;
        }
        long currentTimeInMillis = this.clock.millis();
        long beenInOpenState = currentTimeInMillis - stateLong;
        if (beenInOpenState >= this.waitDurationInOpenStateInMillis) {
            if (this.state.compareAndSet(stateLong, 0x4000000000000001L)) {
                if (logger.isTraceEnabled()) {
                    logger.trace("CircuitBreaker '{}' acquired in HALF_OPEN state with {} calls left", (Object)this.name, (Object)(this.config.permittedCallsInHalfOpenState() - 1));
                }
                this.onStateChange(CircuitBreaker.State.OPEN, CircuitBreaker.State.HALF_OPEN, "acquire", null);
                this.metrics.recordCallAcquire(this.name, CircuitBreakerMetrics.CallAcquireStatus.PERMITTED);
                return true;
            }
            return this.tryAcquire();
        }
        if (logger.isTraceEnabled()) {
            logger.trace("CircuitBreaker '{}' rejected in OPEN state for waiting '{}' when require minimum wait time '{}'", new Object[]{this.name, TimeUtils.durationForLogging((long)beenInOpenState), TimeUtils.durationForLogging((long)this.waitDurationInOpenStateInMillis)});
        }
        this.metrics.recordCallAcquire(this.name, CircuitBreakerMetrics.CallAcquireStatus.REJECTED);
        return false;
    }

    @Override
    public void releaseOnSuccess() {
        long newStateLong;
        long currentStateLong;
        if (Boolean.FALSE.equals(this.config.enabled())) {
            logger.debug("CircuitBreaker '{}' is disabled", (Object)this.name);
            return;
        }
        while (!this.state.compareAndSet(currentStateLong = this.state.get(), newStateLong = this.calculateStateOnSuccess(currentStateLong))) {
        }
        CircuitBreaker.State newState = this.getState(newStateLong);
        CircuitBreaker.State prevState = this.getState(currentStateLong);
        if (prevState != newState) {
            this.onStateChange(prevState, newState, "success", null);
        }
        if (prevState == newState) {
            logger.trace("CircuitBreaker '{}' released in {} state on success", (Object)this.name, (Object)newState);
        } else {
            logger.trace("CircuitBreaker '{}' released from {} to {} state on success", new Object[]{this.name, prevState, newState});
        }
    }

    private long calculateStateOnSuccess(long currentState) {
        CircuitBreaker.State state = this.getState(currentState);
        if (state == CircuitBreaker.State.CLOSED) {
            int total = this.countClosedTotal(currentState) + 1;
            if ((long)total == this.config.slidingWindowSize()) {
                return Long.MIN_VALUE;
            }
            return currentState + 1L;
        }
        if (state == CircuitBreaker.State.HALF_OPEN) {
            int permitted;
            int success = this.countHalfOpenSuccess(currentState) + 1;
            if (success >= (permitted = this.config.permittedCallsInHalfOpenState().intValue())) {
                return Long.MIN_VALUE;
            }
            return currentState + 65536L;
        }
        return currentState;
    }

    @Override
    public void releaseOnError(@Nonnull Throwable throwable) {
        long newStateLong;
        long currentStateLong;
        if (Boolean.FALSE.equals(this.config.enabled())) {
            logger.debug("CircuitBreaker '{}' is disabled", (Object)this.name);
            return;
        }
        if (!this.failurePredicate.test(throwable)) {
            if (logger.isTraceEnabled()) {
                long currentStateLong2 = this.state.get();
                CircuitBreaker.State currentState = this.getState(currentStateLong2);
                logger.trace("CircuitBreaker '{}' skipped error in {} state due to predicate test failed: {}", new Object[]{this.name, currentState, throwable.toString()});
            }
            return;
        }
        while (!this.state.compareAndSet(currentStateLong = this.state.get(), newStateLong = this.calculateStateOnFailure(currentStateLong))) {
        }
        CircuitBreaker.State newState = this.getState(newStateLong);
        CircuitBreaker.State prevState = this.getState(currentStateLong);
        if (prevState != newState) {
            this.onStateChange(prevState, newState, "error", throwable);
        }
        if (prevState == newState) {
            logger.trace("CircuitBreaker '{}' released in {} state on error: {}", new Object[]{this.name, newState, throwable.toString()});
        } else {
            logger.trace("CircuitBreaker '{}' released from {} to {} state on error: {}", new Object[]{this.name, prevState, newState, throwable.toString()});
        }
    }

    private long calculateStateOnFailure(long currentState) {
        CircuitBreaker.State state = this.getState(currentState);
        if (state == CircuitBreaker.State.CLOSED) {
            int total = this.countClosedTotal(currentState) + 1;
            if ((long)total < this.config.minimumRequiredCalls()) {
                return currentState + 0x80000001L;
            }
            float errors = this.countClosedErrors(currentState) + 1;
            int failureRatePercentage = (int)(errors / (float)total * 100.0f);
            if (failureRatePercentage >= this.config.failureRateThreshold()) {
                return this.getOpenState();
            }
            if ((long)total == this.config.slidingWindowSize()) {
                return Long.MIN_VALUE;
            }
            return currentState + 0x80000001L;
        }
        if (state == CircuitBreaker.State.HALF_OPEN) {
            return this.getOpenState();
        }
        return currentState;
    }
}

