package security.whisper.javastix.coo.extension.types;

import com.fasterxml.jackson.annotation.*;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import security.whisper.javastix.coo.extension.CyberObservableExtension;
import security.whisper.javastix.coo.objects.FileCoo;
import security.whisper.javastix.validation.constraints.businessrule.BusinessRule;
import security.whisper.javastix.validation.constraints.coo.allowedparents.AllowedParents;
import security.whisper.javastix.validation.constraints.defaulttypevalue.DefaultTypeValue;
import security.whisper.javastix.validation.groups.DefaultValuesProcessor;
import org.immutables.serial.Serial;
import org.immutables.value.Value;

import java.util.Map;
import java.util.Optional;

import static com.fasterxml.jackson.annotation.JsonInclude.Include.NON_EMPTY;

/**
 * The Raster Image file extension specifies a default extension for capturing
 * properties specific to image files.
 *
 */
@Value.Immutable @Serial.Version(1L)
@DefaultTypeValue(value = "raster-image-ext", groups = {DefaultValuesProcessor.class})
@Value.Style(typeAbstract="*Ext", typeImmutable="*", validationMethod = Value.Style.ValidationMethod.NONE, additionalJsonAnnotations = {JsonTypeName.class}, passAnnotations = {AllowedParents.class}, depluralize = true)
@JsonSerialize(as = RasterImageFileExtension.class) @JsonDeserialize(builder = RasterImageFileExtension.Builder.class)
@JsonInclude(value = NON_EMPTY, content= NON_EMPTY)
@JsonPropertyOrder({ "image_height", "image_width", "bits_per_pixel", "image_compression_algorithm", "exif_tags" })
@JsonTypeName("raster-image-ext")
@AllowedParents({FileCoo.class})
@BusinessRule(ifExp = "true", thenExp = "getImageHeight().isPresent() == true || getImageWidth().isPresent() == true || getBitsPerPixel().isPresent() == false || getImageCompressionAlgorithm().isPresent() == true || getExifTags().isEmpty() == true", errorMessage = "At least 1 field must be used in Raster Image File Extension.")
public interface RasterImageFileExtensionExt extends CyberObservableExtension {

    //@TODO Spec is missing direction about limits: Value likely needs to be MUST be positive
    @JsonProperty("image_height")
    @JsonPropertyDescription("Specifies the height of the image in the image file, in pixels.")
    Optional<Long> getImageHeight();

    //@TODO Spec is missing direction about limits: Value likely needs to be MUST be positive
    @JsonProperty("image_width")
    @JsonPropertyDescription("Specifies the width of the image in the image file, in pixels.")
    Optional<Long> getImageWidth();

    @JsonProperty("bits_per_pixel")
    @JsonPropertyDescription("Specifies the sum of bits used for each color channel in the image in the image file, and thus the total number of pixels used for expressing the color depth of the image.")
    Optional<Long> getBitsPerPixel();

    @JsonProperty("image_compression_algorithm")
    @JsonPropertyDescription("Specifies the name of the compression algorithm used to compress the image in the image file, if applicable.")
    Optional<String> getImageCompressionAlgorithm();

    @JsonProperty("exif_tags")
    @JsonPropertyDescription("Specifies the set of EXIF tags found in the image file, as a dictionary. Each key/value pair in the dictionary represents the name/value of a single EXIF tag.")
    Map<String,Object> getExifTags();

}
