package security.whisper.javastix.graph.sdo;

import security.whisper.javastix.graph.GraphGenerator;
import security.whisper.javastix.graph.elements.GraphElement;
import security.whisper.javastix.graph.elements.Node;
import security.whisper.javastix.sdo.objects.MalwareSdo;

import java.util.HashSet;
import java.util.Set;

/**
 * Specialized graph generator for Malware SDO.
 * Adds malware specific metadata to graph nodes.
 */
public class MalwareGraphGenerator implements GraphGenerator {

    private final MalwareSdo malware;

    public MalwareGraphGenerator(MalwareSdo malware) {
        this.malware = malware;
    }

    @Override
    public Set<GraphElement> process() {
        Set<GraphElement> elements = new HashSet<>();
        elements.add(generateNode());
        return elements;
    }

    private Node generateNode() {
        Node node = new Node(malware.getId(), "malware", null, malware);

        // Add malware specific properties
        node.getData().setNodeLabel(malware.getName());
        node.getData().setNodeColor("#FF9800"); // Deep orange for malware
        node.getData().setNodeShape("triangle");
        node.getData().setNodeIcon("malware");

        // Add additional metadata
        node.getData().getAdditionalProperties().put("name", malware.getName());

        if (malware.getDescription() != null) {
            node.getData().getAdditionalProperties().put("description",
                malware.getDescription().orElse(""));
        }

        if (malware.getLabels() != null && !malware.getLabels().isEmpty()) {
            node.getData().getAdditionalProperties().put("labels", malware.getLabels());

            // Determine malware category from labels
            String category = "unknown";
            for (String label : malware.getLabels()) {
                if (label.contains("ransomware")) {
                    category = "ransomware";
                    node.getData().setNodeColor("#D32F2F"); // Darker red for ransomware
                    break;
                } else if (label.contains("trojan")) {
                    category = "trojan";
                } else if (label.contains("worm")) {
                    category = "worm";
                } else if (label.contains("rootkit")) {
                    category = "rootkit";
                } else if (label.contains("backdoor")) {
                    category = "backdoor";
                }
            }
            node.getData().getAdditionalProperties().put("malware_category", category);
        }

        if (malware.getKillChainPhases() != null && !malware.getKillChainPhases().isEmpty()) {
            node.getData().getAdditionalProperties().put("kill_chain_phases",
                malware.getKillChainPhases());
        }

        return node;
    }
}