package security.whisper.javastix.sdo.objects;

import com.fasterxml.jackson.annotation.*;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import security.whisper.javastix.common.StixInstant;
import security.whisper.javastix.redaction.Redactable;
import security.whisper.javastix.sdo.DomainObject;
import security.whisper.javastix.validation.constraints.defaulttypevalue.DefaultTypeValue;
import security.whisper.javastix.validation.constraints.vocab.Vocab;
import security.whisper.javastix.validation.groups.DefaultValuesProcessor;
import security.whisper.javastix.vocabulary.vocabularies.MalwareResult;
import org.immutables.serial.Serial;
import org.immutables.value.Value;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.Optional;
import java.util.Set;

import static com.fasterxml.jackson.annotation.JsonInclude.Include.NON_EMPTY;

/**
 * malware-analysis
 * <p>
 * A Malware Analysis captures the results of a particular analysis performed on a malware instance or family.
 * Each result captures the analysis product (including version) that was used to analyze the malware along with
 * the result of that analysis.
 */
@Value.Immutable @Serial.Version(1L)
@JsonTypeName("malware-analysis")
@DefaultTypeValue(value = "malware-analysis", groups = {DefaultValuesProcessor.class})
@Value.Style(typeAbstract="*Sdo", typeImmutable="*", validationMethod = Value.Style.ValidationMethod.NONE, additionalJsonAnnotations = {JsonTypeName.class}, depluralize = true)
@JsonSerialize(as = MalwareAnalysis.class) @JsonDeserialize(builder = MalwareAnalysis.Builder.class)
@JsonPropertyOrder({"type", "spec_version", "id", "created_by_ref", "created",
        "modified", "revoked", "labels", "confidence", "external_references",
        "object_marking_refs", "granular_markings", "product", "version",
        "host_vm_ref", "operating_system_ref", "installed_software_refs",
        "configuration_version", "module", "analysis_engine_version",
        "analysis_definition_version", "submitted", "analysis_started",
        "analysis_ended", "av_result", "analysis_sco_refs", "sample_ref"})
@Redactable
public interface MalwareAnalysisSdo extends DomainObject {

    @NotBlank
    @JsonProperty("product")
    @JsonPropertyDescription("The name of the analysis product that was used to perform the analysis.")
    @Redactable
    String getProduct();

    @JsonProperty("version") @JsonInclude(value = NON_EMPTY, content= NON_EMPTY)
    @JsonPropertyDescription("The version of the analysis product that was used to perform the analysis.")
    @Redactable
    Optional<String> getVersion();

    @JsonProperty("host_vm_ref") @JsonInclude(value = NON_EMPTY, content= NON_EMPTY)
    @JsonPropertyDescription("A reference to a SCO Software object for the VM that hosted the malware during the analysis.")
    @Redactable
    Optional<String> getHostVmRef();

    @JsonProperty("operating_system_ref") @JsonInclude(value = NON_EMPTY, content= NON_EMPTY)
    @JsonPropertyDescription("A reference to a SCO Software object for the operating system used for the analysis VM.")
    @Redactable
    Optional<String> getOperatingSystemRef();

    @JsonProperty("installed_software_refs") @JsonInclude(value = NON_EMPTY, content= NON_EMPTY)
    @JsonPropertyDescription("A list of references to SCO Software objects for the operating system software installed on the analysis VM.")
    @Redactable
    Set<String> getInstalledSoftwareRefs();

    @JsonProperty("configuration_version") @JsonInclude(value = NON_EMPTY, content= NON_EMPTY)
    @JsonPropertyDescription("The version of the configuration used for the analysis.")
    @Redactable
    Optional<String> getConfigurationVersion();

    @JsonProperty("module") @JsonInclude(value = NON_EMPTY, content= NON_EMPTY)
    @JsonPropertyDescription("The name of the specific analysis module that was used, if different from the product.")
    @Redactable
    Optional<String> getModule();

    @JsonProperty("analysis_engine_version") @JsonInclude(value = NON_EMPTY, content= NON_EMPTY)
    @JsonPropertyDescription("The version of the analysis engine or runtime environment.")
    @Redactable
    Optional<String> getAnalysisEngineVersion();

    @JsonProperty("analysis_definition_version") @JsonInclude(value = NON_EMPTY, content= NON_EMPTY)
    @JsonPropertyDescription("The version of the analysis definitions used (e.g., virus definition version).")
    @Redactable
    Optional<String> getAnalysisDefinitionVersion();

    @JsonProperty("submitted") @JsonInclude(value = NON_EMPTY, content= NON_EMPTY)
    @JsonPropertyDescription("The date and time that the malware sample was submitted for analysis.")
    @Redactable
    Optional<StixInstant> getSubmitted();

    @JsonProperty("analysis_started") @JsonInclude(value = NON_EMPTY, content= NON_EMPTY)
    @JsonPropertyDescription("The date and time that the malware analysis started.")
    @Redactable
    Optional<StixInstant> getAnalysisStarted();

    @JsonProperty("analysis_ended") @JsonInclude(value = NON_EMPTY, content= NON_EMPTY)
    @JsonPropertyDescription("The date and time that the malware analysis ended.")
    @Redactable
    Optional<StixInstant> getAnalysisEnded();

    @JsonProperty("av_result") @JsonInclude(value = NON_EMPTY, content= NON_EMPTY)
    @JsonPropertyDescription("The result of the anti-virus analysis. Open vocabulary - malware-av-result-ov")
    @Vocab(MalwareResult.class)
    @Redactable
    Optional<String> getAvResult();

    @JsonProperty("analysis_sco_refs") @JsonInclude(value = NON_EMPTY, content= NON_EMPTY)
    @JsonPropertyDescription("A list of references to SCOs that were captured during the analysis process.")
    @Redactable
    Set<String> getAnalysisScoRefs();

    @JsonProperty("sample_ref") @JsonInclude(value = NON_EMPTY, content= NON_EMPTY)
    @JsonPropertyDescription("A reference to the SCO file, malware or artifact object that was analyzed.")
    @Redactable
    Optional<String> getSampleRef();

}