package security.whisper.javastix.sdo.objects;

import com.fasterxml.jackson.annotation.*;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import security.whisper.javastix.redaction.Redactable;
import security.whisper.javastix.sdo.DomainObject;
import security.whisper.javastix.sdo.types.KillChainPhaseType;
import security.whisper.javastix.validation.constraints.defaulttypevalue.DefaultTypeValue;
import security.whisper.javastix.validation.constraints.vocab.Vocab;
import security.whisper.javastix.validation.groups.DefaultValuesProcessor;
import security.whisper.javastix.vocabulary.vocabularies.MalwareLabels;
import org.hibernate.validator.constraints.Length;
import org.immutables.serial.Serial;
import org.immutables.value.Value;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;
import java.util.Collections;
import java.util.Optional;
import java.util.Set;

import static com.fasterxml.jackson.annotation.JsonInclude.Include.NON_EMPTY;


/**
 * malware
 * <p>
 * Malware is a type of TTP that is also known as malicious code and malicious software, refers to a program that is inserted into a system,
 * usually covertly, with the intent of compromising the confidentiality, integrity, or availability of the victim's data, applications,
 * or operating system (OS) or of otherwise annoying or disrupting the victim.
 */
@Value.Immutable @Serial.Version(1L)
@DefaultTypeValue(value = "malware", groups = {DefaultValuesProcessor.class})
@Value.Style(typeAbstract="*Sdo", typeImmutable="*", validationMethod = Value.Style.ValidationMethod.NONE, additionalJsonAnnotations = {JsonTypeName.class}, depluralize = true)
@JsonTypeName("malware")
@JsonSerialize(as = Malware.class) @JsonDeserialize(builder = Malware.Builder.class)
@JsonPropertyOrder({"type", "id", "created_by_ref", "created",
        "modified", "revoked", "labels", "external_references",
        "object_marking_refs", "granular_markings", "name", "description",
        "kill_chain_phases"})
@Redactable
public interface MalwareSdo extends DomainObject {

    @Override
    @Value.Default
    @Vocab(MalwareLabels.class)
    @JsonPropertyDescription("The type of malware being described. Open Vocab - malware-label-ov")
    @NotNull
    @Size(min = 1, message = "At least one label from malware-label-ov must be used")
    @Redactable(useMask = true)
    default Set<@Length(min = 1) String> getLabels() {
        return Collections.emptySet();
    }

    @NotBlank
    @JsonProperty("name")
    @JsonPropertyDescription("The name used to identify the Malware.")
    @Redactable(useMask = true)
    String getName();

    @JsonProperty("description") @JsonInclude(value = NON_EMPTY, content= NON_EMPTY)
    @JsonPropertyDescription("Provides more context and details about the Malware object.")
    @Redactable
    Optional<String> getDescription();

    @NotNull
    @Value.Default
    @JsonProperty("kill_chain_phases")
    @JsonInclude(NON_EMPTY)
    @JsonPropertyDescription("The list of kill chain phases for which this Malware instance can be used.")
    @Redactable
    default Set<KillChainPhaseType> getKillChainPhases() {
        return Collections.emptySet();
    }

}
