package security.whisper.javastix.vocabulary.vocabularies;

import com.fasterxml.jackson.annotation.JsonProperty;
import security.whisper.javastix.vocabulary.StixVocabulary;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Malware Capabilities is an open vocabulary that describes capabilities that a piece of malware may possess.
 * These capabilities are high-level descriptions of what the malware can do.
 */
public class MalwareCapabilities implements StixVocabulary {

    @JsonProperty("malware_capabilities_vocabulary")
    private Set<String> terms = new HashSet<>(Arrays.asList(
            "anti-debugging",              // Techniques to detect/prevent debugging
            "anti-disassembly",            // Techniques to prevent disassembly
            "anti-emulation",              // Techniques to detect/prevent emulation
            "anti-memory-forensics",       // Techniques to prevent memory forensics
            "anti-sandbox",                // Techniques to detect/evade sandboxes
            "anti-vm",                     // Techniques to detect virtual machines
            "captures-input",              // Captures user input (keylogging, etc.)
            "cleans-traces-of-infection",  // Removes evidence of infection
            "commits-fraud",               // Performs fraudulent activities
            "communicates-with-c2",        // Communicates with command and control
            "compromises-data-availability", // Affects data availability
            "compromises-data-integrity",  // Modifies or corrupts data
            "compromises-system-availability", // Affects system availability
            "controls-local-machine",      // Controls the infected machine
            "degrades-security-software",  // Disables or degrades security software
            "degrades-system-updates",     // Prevents system updates
            "determines-c2-server",        // Determines C2 server dynamically
            "emails-spam",                 // Sends spam emails
            "escalates-privileges",        // Escalates privileges
            "evades-av",                   // Evades antivirus detection
            "exfiltrates-data",            // Steals and sends data
            "fingerprints-host",           // Gathers host information
            "hides-artifacts",             // Hides files, processes, etc.
            "hides-executing-code",        // Hides executing code
            "infects-files",               // Infects other files
            "infects-remote-machines",     // Spreads to other machines
            "installs-other-components",   // Downloads/installs additional components
            "persists-after-reboot",       // Maintains persistence
            "prevents-artifact-access",    // Prevents access to its artifacts
            "prevents-artifact-deletion",  // Prevents deletion of its artifacts
            "probes-network-configuration", // Discovers network configuration
            "self-modifies",               // Modifies its own code
            "steals-authentication-credentials", // Steals credentials
            "violates-system-operational-integrity" // Violates system integrity
    ));

    @Override
    public Set<String> getAllTerms() {
        return terms;
    }

    @Override
    public Set<String> getAllTermsWithAdditional(String[] terms) {
        return Stream.concat(getAllTerms().stream(), Arrays.stream(terms))
                .collect(Collectors.toCollection(HashSet::new));
    }
}