/*
 * Decompiled with CFR 0.152.
 */
package security.whisper.javastix.pattern;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import security.whisper.javastix.pattern.PatternExpression;
import security.whisper.javastix.pattern.StixPattern;
import security.whisper.javastix.pattern.StixPatternParser;
import security.whisper.javastix.pattern.expressions.ComparisonExpression;
import security.whisper.javastix.pattern.expressions.CompoundExpression;
import security.whisper.javastix.pattern.expressions.ListExpression;
import security.whisper.javastix.pattern.expressions.ObjectPathExpression;
import security.whisper.javastix.pattern.expressions.ObservationExpression;
import security.whisper.javastix.pattern.expressions.QualifiedExpression;

public class StixPatternValidator {
    private static final Set<String> VALID_OBSERVABLE_TYPES = new HashSet<String>(Arrays.asList("artifact", "autonomous-system", "directory", "domain-name", "email-addr", "email-message", "file", "ipv4-addr", "ipv6-addr", "mac-addr", "mutex", "network-traffic", "process", "software", "url", "user-account", "windows-registry-key", "x509-certificate", "x-509-certificate"));
    private static final Set<String> VALID_OPERATORS = new HashSet<String>(Arrays.asList("=", "!=", ">", "<", ">=", "<=", "IN", "LIKE", "MATCHES", "ISSUBSET", "ISSUPERSET"));

    public static ValidationResult validate(String patternString) {
        if (patternString == null || patternString.trim().isEmpty()) {
            return new ValidationResult(false, "Pattern cannot be null or empty");
        }
        StixPattern pattern = StixPatternParser.parse(patternString);
        if (!pattern.isValid()) {
            return new ValidationResult(false, pattern.getErrorMessage().orElse("Failed to parse pattern"));
        }
        ArrayList<String> errors = new ArrayList<String>();
        StixPatternValidator.validateExpression(pattern.getExpression(), errors);
        if (!errors.isEmpty()) {
            return new ValidationResult(false, String.join((CharSequence)"; ", errors));
        }
        return new ValidationResult(true, "Pattern is valid");
    }

    private static void validateExpression(PatternExpression expr, List<String> errors) {
        if (expr == null) {
            errors.add("Null expression found");
            return;
        }
        switch (expr.getType()) {
            case OBSERVATION: {
                StixPatternValidator.validateObservation((ObservationExpression)expr, errors);
                break;
            }
            case COMPARISON: {
                StixPatternValidator.validateComparison((ComparisonExpression)expr, errors);
                break;
            }
            case COMPOUND: {
                StixPatternValidator.validateCompound((CompoundExpression)expr, errors);
                break;
            }
            case QUALIFIED: {
                StixPatternValidator.validateQualified((QualifiedExpression)expr, errors);
                break;
            }
            case OBJECT_PATH: {
                StixPatternValidator.validateObjectPath((ObjectPathExpression)expr, errors);
                break;
            }
            case LITERAL: {
                break;
            }
            case LIST: {
                StixPatternValidator.validateList((ListExpression)expr, errors);
                break;
            }
            default: {
                errors.add("Unknown expression type: " + (Object)((Object)expr.getType()));
            }
        }
    }

    private static void validateObservation(ObservationExpression obs, List<String> errors) {
        if (obs.getComparison() == null) {
            errors.add("Observation must contain a comparison");
        } else {
            StixPatternValidator.validateExpression(obs.getComparison(), errors);
        }
    }

    private static void validateComparison(ComparisonExpression comp, List<String> errors) {
        if (!VALID_OPERATORS.contains(comp.getOperator())) {
            errors.add("Invalid operator: " + comp.getOperator());
        }
        if (comp.getLeft() == null) {
            errors.add("Comparison left side cannot be null");
        } else if (comp.getLeft().getType() != PatternExpression.ExpressionType.OBJECT_PATH) {
            errors.add("Comparison left side must be an object path");
        } else {
            StixPatternValidator.validateExpression(comp.getLeft(), errors);
        }
        if (comp.getRight() == null) {
            errors.add("Comparison right side cannot be null");
        } else {
            StixPatternValidator.validateExpression(comp.getRight(), errors);
        }
    }

    private static void validateCompound(CompoundExpression compound, List<String> errors) {
        if (compound.getLeft() == null || compound.getRight() == null) {
            errors.add("Compound expression must have both left and right expressions");
        } else {
            StixPatternValidator.validateExpression(compound.getLeft(), errors);
            StixPatternValidator.validateExpression(compound.getRight(), errors);
        }
    }

    private static void validateQualified(QualifiedExpression qual, List<String> errors) {
        if (qual.getObservation() == null) {
            errors.add("Qualified expression must have an observation");
        } else {
            StixPatternValidator.validateExpression(qual.getObservation(), errors);
        }
        switch (qual.getQualifier()) {
            case WITHIN: {
                StixPatternValidator.validateTimeWindow(qual.getQualifierValue(), errors);
                break;
            }
            case REPEATS: {
                StixPatternValidator.validateRepeatCount(qual.getQualifierValue(), errors);
                break;
            }
            case START: 
            case STOP: {
                StixPatternValidator.validateTimestamp(qual.getQualifierValue(), errors);
            }
        }
    }

    private static void validateObjectPath(ObjectPathExpression path, List<String> errors) {
        if (path.getObjectType() != null && !VALID_OBSERVABLE_TYPES.contains(path.getObjectType())) {
            errors.add("Invalid observable type: " + path.getObjectType());
        }
        if (path.getPropertyPath() == null || path.getPropertyPath().isEmpty()) {
            errors.add("Property path cannot be empty");
        }
    }

    private static void validateList(ListExpression list, List<String> errors) {
        if (list.getValues() == null || list.getValues().isEmpty()) {
            errors.add("List cannot be empty");
        } else {
            for (PatternExpression value : list.getValues()) {
                StixPatternValidator.validateExpression(value, errors);
            }
        }
    }

    private static void validateTimeWindow(String timeWindow, List<String> errors) {
        if (timeWindow == null || !timeWindow.matches("\\d+\\s+(SECONDS?|MINUTES?|HOURS?|DAYS?)")) {
            errors.add("Invalid time window format: " + timeWindow);
        }
    }

    private static void validateRepeatCount(String repeatCount, List<String> errors) {
        try {
            int count = Integer.parseInt(repeatCount);
            if (count < 1) {
                errors.add("Repeat count must be positive: " + repeatCount);
            }
        }
        catch (NumberFormatException e) {
            errors.add("Invalid repeat count: " + repeatCount);
        }
    }

    private static void validateTimestamp(String timestamp, List<String> errors) {
        if (timestamp == null || !timestamp.matches("t'\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}(\\.\\d{3})?Z'")) {
            errors.add("Invalid timestamp format: " + timestamp);
        }
    }

    public static class ValidationResult {
        private final boolean valid;
        private final String message;

        public ValidationResult(boolean valid, String message) {
            this.valid = valid;
            this.message = message;
        }

        public boolean isValid() {
            return this.valid;
        }

        public String getMessage() {
            return this.message;
        }
    }
}

