package to.then.kie;

import com.amazonaws.services.lambda.AWSLambda;
import com.amazonaws.services.lambda.AWSLambdaClient;
import com.amazonaws.services.lambda.runtime.Context;
import com.amazonaws.services.lambda.runtime.events.KinesisEvent;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.AmazonS3URI;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.sqs.AmazonSQS;
import com.amazonaws.services.sqs.AmazonSQSClient;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import javax.enterprise.inject.InjectionException;
import javax.inject.Inject;
import javax.ws.rs.client.Client;
import org.apache.commons.io.IOUtils;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.log4j.Logger;
import org.drools.compiler.builder.impl.KnowledgeBuilderImpl;
import org.drools.compiler.kie.builder.impl.AbstractKieModule;
import org.drools.compiler.kie.builder.impl.KieBuilderImpl;
import org.eclipse.aether.repository.RemoteRepository;
import org.jboss.resteasy.client.jaxrs.ResteasyClientBuilder;
import org.jboss.resteasy.client.jaxrs.engines.ApacheHttpClient4Engine;
import org.jboss.resteasy.plugins.providers.jackson.ResteasyJackson2Provider;
import org.kie.api.KieBase;
import org.kie.api.KieServices;
import org.kie.api.builder.KieBuilder;
import org.kie.api.builder.KieFileSystem;
import org.kie.api.builder.Message.Level;
import org.kie.api.builder.ReleaseId;
import org.kie.api.builder.model.KieBaseModel;
import org.kie.api.builder.model.KieModuleModel;
import org.kie.api.builder.model.KieSessionModel;
import org.kie.api.runtime.KieContainer;
import org.kie.api.runtime.StatelessKieSession;
import org.kie.scanner.embedder.MavenSettings;
import to.then.kie.model.Module;
import to.then.kie.model.Settings;

public class KieFunction {

    private final static Logger log = Logger.getLogger(KieFunction.class);

    private final AmazonS3 s3;
    private final ObjectMapper json;
    private final KieServices kie;
    private final Map<Class, Object> providers;
    private final KieRepository repository;

    public KieFunction(KieRepository repository, AmazonS3 s3, AWSLambda lambda, Client http) {

        this.repository = repository;
        json = new ObjectMapper()
                .findAndRegisterModules()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        this.s3 = s3;
        kie = KieServices.Factory.get();
        providers = new LinkedHashMap();
        providers.put(AWSLambda.class, lambda);
        providers.put(AmazonS3.class, new AmazonS3Client());
        providers.put(AmazonSQS.class, new AmazonSQSClient());
        providers.put(Client.class, http);
        providers.put(ObjectMapper.class, json);
        providers.put(Logger.class, log);

        unpackRepository("/maven.zip");
    }

    public KieFunction(AmazonS3 s3, AWSLambda lambda, Client http) {
        this(new KieRepository(s3), s3, lambda, http);
    }

    public KieFunction() {
        this(new AmazonS3Client(), new AWSLambdaClient(), new ResteasyClientBuilder().httpEngine(
                new ApacheHttpClient4Engine(HttpClients.custom()
                        .setConnectionManager(new PoolingHttpClientConnectionManager())
                        .build()))
                .register(ResteasyJackson2Provider.class).build());
    }

    public void apply(KinesisEvent event, Context context) {
        try {
            log.info(json.writeValueAsString(event));
        } catch (JsonProcessingException ex) {
        }
        event.getRecords().forEach((record) -> {
            try {
                JsonNode request = json.readTree(record.getKinesis().getData().array());
                AmazonS3URI settingsUri = new AmazonS3URI(request.get("settings").textValue());
                Settings settings = getSettings(settingsUri);
                List<String> profiles = null;
                if (request.has("profiles")) {
                    profiles = json.convertValue(request.get("profiles"), List.class);
                }
                String command = request.get("command").textValue();
                ObjectNode result = json.createObjectNode();
                Set<RemoteRepository> repositories = settings.getRemoteRepositories(profiles);
                Properties properties = settings.getProperties(profiles);
                Map<String, String> aliases = settings.getAliases(profiles);
                String releaseId = request.get("releaseId").textValue();
                releaseId = aliases.getOrDefault(releaseId, releaseId);
                result.put("status", "ok");
                switch (command) {
                    case "deploy":
                        Module module = json.convertValue(request.get("module"), Module.class);
                        KieBuilder builder = deploy(releaseId(releaseId), module, repositories);
                        if (builder.getResults().hasMessages(Level.ERROR)) {
                            result.put("status", "error");
                        }
                        if (!builder.getResults().getMessages().isEmpty()) {
                            result.set("messages", json.convertValue(builder.getResults().getMessages(), ArrayNode.class));
                        }
                        break;
                    case "insert":
                        ArrayNode inserts = (ArrayNode) request.get("objects");
                        insert(releaseId(releaseId), inserts, properties, repositories);
                        break;
                }
                log.info(json.writeValueAsString(result));
            } catch (Exception ex) {
                log.error(ex.getMessage(), ex);
            }
        });
    }

    private Settings getSettings(AmazonS3URI uri) {
        S3Object s3Object = s3.getObject(uri.getBucket(), uri.getKey());
        Settings settings = null;
        try (InputStream objectIn = s3Object.getObjectContent()) {
            settings = json.readValue(objectIn, Settings.class);
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
        return settings;
    }

    protected KieBuilder deploy(ReleaseId releaseId, Module module, Collection<RemoteRepository> repositories) {
        KieBuilder builder = build(releaseId, module, repositories);
        if (!builder.getResults().hasMessages(Level.ERROR)) {
            repository.putKieModule((AbstractKieModule) builder.getKieModule(), module.getDistribution().getRepository().toRemoteRepository());
        } else {
            log.error(builder.getResults().getMessages(Level.ERROR));
        }
        return builder;
    }
    
    protected KieBuilder build(ReleaseId releaseId, Module module, Collection<RemoteRepository> repositories) {
        List<ReleaseId> dependencyIds = new LinkedList();
        if (module.getDependencies() != null) {
            module.getDependencies().forEach((dependency) -> {
                repository.resolveDependencies(dependency, repositories);
                ReleaseId dependencyId = releaseId(dependency);
                dependencyIds.add(dependencyId);
            });
        }
        KieFileSystem files = kie.newKieFileSystem();
        KieModuleModel kModuleModel = configureKieModuleModel(kie.newKieModuleModel(), module.getName(), releaseId, dependencyIds, module.getIncludes());
        files.writeKModuleXML(kModuleModel.toXML());
        files.writePomXML(getPomXml(releaseId, dependencyIds));
        if (module.getResources() != null) {
            module.getResources().forEach((path, resource) -> {
                files.write("src/main/resources/" + path, resource);
            });
        }
        KieBuilderImpl builder = (KieBuilderImpl) kie.newKieBuilder(files);
        builder.getPomModel().getDependencies().addAll(dependencyIds);
        builder.buildAll();
        return builder;
    }

    protected StatelessKieSession newStatelessKieSession(Properties properties, AbstractKieModule kmodule, KieContainer kcontainer) {
        StatelessKieSession session = kcontainer.newStatelessKieSession();
        KnowledgeBuilderImpl kb = (KnowledgeBuilderImpl) kmodule.getKnowledgeBuilderForKieBase(kmodule.getKieModuleModel().getKieBaseModels().keySet().iterator().next());
        kb.getGlobals().forEach((String key, Class<?> value) -> {
            if (providers.containsKey(value)) {
                session.setGlobal(key, providers.get(value));
            } else if (Properties.class.equals(value)) {
                session.setGlobal(key, properties);
            } else {
                try {
                    Class type = kcontainer.getClassLoader().loadClass(value.getName());
                    for (Constructor constructor : type.getConstructors()) {
                        if (constructor.getAnnotation(Inject.class) != null) {
                            try {
                                Object[] args = new Object[constructor.getParameterCount()];
                                for (int index = 0; index < constructor.getParameterCount(); index++) {
                                    Class parameterType = constructor.getParameterTypes()[index];
                                    if (providers.get(parameterType) != null) {
                                        args[index] = providers.get(parameterType);
                                    } else if (Properties.class.equals(parameterType)) {
                                       args[index] = properties;
                                    } else {
                                        throw new InjectionException(type.getName() + ": " + parameterType);
                                    }
                                }
                                session.setGlobal(key, constructor.newInstance(args));
                            } catch (InstantiationException | IllegalAccessException | IllegalArgumentException | InvocationTargetException ex) {
                                throw new RuntimeException(ex);
                            }
                        }
                    }
                } catch (ClassNotFoundException ex) {
                    throw new RuntimeException(ex);
                }
            }
        });
        return session;
    }

    protected String getPomXml(ReleaseId releaseId, List<ReleaseId> dependencies) {
        String pom
                = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
                + "<project xmlns=\"http://maven.apache.org/POM/4.0.0\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n"
                + "         xsi:schemaLocation=\"http://maven.apache.org/POM/4.0.0 http://maven.apache.org/maven-v4_0_0.xsd\">\n"
                + "  <modelVersion>4.0.0</modelVersion>\n"
                + "\n"
                + "  <groupId>" + releaseId.getGroupId() + "</groupId>\n"
                + "  <artifactId>" + releaseId.getArtifactId() + "</artifactId>\n"
                + "  <version>" + releaseId.getVersion() + "</version>\n"
                + "\n";
        if (dependencies != null && dependencies.size() > 0) {
            pom += "<dependencies>\n";
            for (ReleaseId dep : dependencies) {
                pom += "<dependency>\n";
                pom += "  <groupId>" + dep.getGroupId() + "</groupId>\n";
                pom += "  <artifactId>" + dep.getArtifactId() + "</artifactId>\n";
                pom += "  <version>" + dep.getVersion() + "</version>\n";
                pom += "</dependency>\n";
            }
            pom += "</dependencies>\n";
        }
        pom += "</project>";
        return pom;
    }

    protected void insert(ReleaseId releaseId, ArrayNode inserts, Properties properties, Set<RemoteRepository> repositories) {
        log.info(releaseId.toExternalForm());
        log.info(inserts);
        AbstractKieModule kmodule = repository.getKieModule(releaseId, repositories);
        KieContainer kcontainer = kie.newKieContainer(kmodule.getReleaseId());
        StatelessKieSession session = newStatelessKieSession(properties, kmodule, kcontainer);
        List objects = new LinkedList();
        inserts.elements().forEachRemaining((node) -> {
            node.fieldNames().forEachRemaining((className) -> {
                try {

                    Object insert = json.convertValue(node.get(className), kcontainer.getClassLoader().loadClass(className));
                    objects.add(insert);
                } catch (ClassNotFoundException ex) {
                    throw new RuntimeException(ex);
                }
            });
        });
        session.execute(kie.getCommands().newInsertElements(objects));
    }

    private ReleaseId releaseId(String releaseId) {
        String[] split = releaseId.split(":");
        return kie.newReleaseId(split[0], split[1], split[2]);
    }

    protected static KieModuleModel configureKieModuleModel(KieModuleModel kModuleModel, String name, ReleaseId releaseId, Collection<ReleaseId> dependencies, Collection<String> includes) {
        KieBaseModel kieBaseModel = kModuleModel
                .newKieBaseModel(name)
                .setDefault(true);
        if (includes != null) {
            includes.forEach((include) -> {
                kieBaseModel.addInclude(include);
            });
        }
        kieBaseModel.newKieSessionModel(name + ".stateless").setType(KieSessionModel.KieSessionType.STATELESS).setDefault(true);
        KieBuilderImpl.setDefaultsforEmptyKieModule(kModuleModel);
        return kModuleModel;
    }

    private void unpackRepository(String resourceName) {
        try {
            try (ZipInputStream stream = new ZipInputStream(KieFunction.class.getResourceAsStream(resourceName))) {
                String outdir = MavenSettings.getSettings().getLocalRepository();
                ZipEntry entry;
                while ((entry = stream.getNextEntry()) != null) {

                    if (entry.getSize() == 0) {
                        File outpath = new File(outdir + "/" + entry.getName());
                        outpath.mkdirs();
                    } else {
                        File outpath = new File(outdir + "/" + entry.getName());
                        outpath.createNewFile();
                        try (FileOutputStream output = new FileOutputStream(outpath)) {
                            IOUtils.copy(stream, output);
                        }
                    }
                }
            }
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

}
